#!/usr/bin/perl -w
# $Id$

use Test;
use strict;
use lib './lib';

BEGIN {plan tests => 7}

use Digest::Perl::MD5 qw(md5 md5_hex md5_base64);

# 1 Testsuite
if (1) {
	print "Trying md5_hex on test suite...\n";
	ok( 
		md5_hex('') eq 'd41d8cd98f00b204e9800998ecf8427e' and
		md5_hex('a') eq '0cc175b9c0f1b6a831c399e269772661' and
		md5_hex('abc') eq '900150983cd24fb0d6963f7d28e17f72' and
		md5_hex('message digest') eq 'f96b697d7cb7938d525a2f31aaf161d0' and
		md5_hex('abcdefghijklmnopqrstuvwxyz') eq
			'c3fcd3d76192e4007dfb496cca67e13b' and
		md5_hex('ABCDEFGHIJKLMNOPQRSTUVWXYZabcdefghijklmnopqrstuvwxyz0123456789') eq
			'd174ab98d277d9f5a5611c2c9f419d9f' and
		md5_hex('12345678901234567890123456789012345678901234567890123456789012345678901234567890') eq
			'57edf4a22be3c955ac49da2e2107b67a'
	);
}

# 2 md5_base64
if (1) {
	print "Trying md5_base64...\n";
	ok ( 
		md5_base64('delta' x 23) eq 'RzlmC2a3rRVNgaZrwusL0Q' and
		md5_base64('carmen' x 26) eq 'WVM3kMiFLRPPRMOo7DQr2w' and
		md5_base64('imperia' x 42) eq 'IjqzkaH6J3rDdQWHuiWuXg'
	);
}

# 3 d-test:
if (1) {
	print "Running d-test\n";
	my $err;
	while(<DATA>) {
		chomp;
		my ($plain,$cipher) = split/:/;
		$err++ if md5_hex($plain) ne $cipher;
	}
	ok(!$err);
}

# 4 Object
if (1) {
	print "Testing MD5-Object...\n";
	my $c = new Digest::Perl::MD5;
	$c->add('XdeltaX');
	ok(
		$c->b64digest eq 'hLA/iI1q1iIKz+uffnsN6w' and
		$c->add('YdeltaY')->hexdigest eq md5_hex('YdeltaY')
	);
}

# 5 Reset
if (1) {
	print "  - reset ";
	my $c = new Digest::Perl::MD5;
	$c->add('foo'); $c->reset(); $c->add('bar');
	ok( $c->hexdigest eq md5_hex('bar') );
}

# 5 Cloning
if (1) {
	print "  - clone ";
	my $c = new Digest::Perl::MD5;
	$c->add('YCarmenY'); my $d = $c->clone(); $c->add('ZKleinZ');
	# use Data::Dumper; print Dumper($c,$d);
	ok ( $c->b64digest eq $d->add('ZKleinZ')->b64digest );
}

# 6 Adding
if (1) {
	print "  - add ";
	my $c = new Digest::Perl::MD5;
	my $anz = 256;
	$c->add($_) for (0..$anz);
	my $string = 'x'x35701;
	ok ( 
		$c->b64digest eq md5_base64( 0..$anz ) and
		$c->add($string)->hexdigest eq md5_hex( $string )
	);
}

# 7 Addfile
if (1) {
	print "  - addfile ";
	my $c = new Digest::Perl::MD5;
	open FILE, './rand.f' or die $!;
	binmode FILE;
	# print $c->addfile(*FILE)->clone->hexdigest,"\n"; # DEBUG
	ok ( $c->addfile(*FILE)->hexdigest eq '2cc2c8b037522964694985b3f9c88b1e' );
	close FILE;
}

# 8 add_bits
if (1) {
	print "  - add_bits ";
	my $c = new Digest::Perl::MD5;
	eval { $c->add_bits('', 9) };
	ok (
		$@ and 
		$c->add_bits('011000010110001001100011')->digest eq md5('abc') and
		$c->add_bits('def', 16)->digest eq md5('de')
	);
}

# 9 Speed-Test
if (1) {
	print "Speed-Test (please be patient)...\n";
	my $count = $ENV{MD5_SPEED_TEST} || 50_000;
	my $t1 = time;
	for (1..$count) { md5('delta') } # encode 64Byte blocks
	my $t2 = time;
	if ($t1 == $t2) {
		print "Ok, you computer is fast. Please increase \$count in $0.\n";
	} else {
		printf "%d blocks took %ds => %.2f blocks/second.\n",
		$count, $t2-$t1, $count/($t2-$t1);
	}
}

__DATA__
m:6f8f57715090da2632453988d9a1501b
Ep:f6508f60f9070dd321948e2e29d54f69
BN7:31c65bc01e31497cf7de537372b5188d
ESEb:be45486e7f9c2dd6aff1311df675f535
EVoKU:623371d815d755fd4b5131a9beb64155
gDhZup:bb5b2909729df42d2c03b9b3784ee349
VqiJ8UB:95171d391a28e77f7862444b56df8ccb
sQFElV6Y:3eeeab59c0c872370cd5267f36c502a2
TAHnCc8QM:1b36853188b600131d0aaae2d7c0ea99
SWf0LAfxRo:23a52e893ea73289b9f674ec82a0b3b5
7P8y8P4D0Qz:951116f3ce64c1cb14753c287b4cdfd0
Pz0mMtoLa0tW:06f33a80c9fc86dba3267c1860d1c3fa
6kywpWeETUCiS:30860356ca296f808f897745f8eb3a58
hdl8TLNjC0VVew:662690a89025445fad59c53146718ade
V5QI107ogTsKDcd:3de369873bcbc4bfc71580b2cb6f9124
MUlP6jzHXI9NuVSR:cea412e61a5b8f322702663e88ee5d8f
GNxeFobTvVc6oe90Z:162d2c130202a32faa119e4d72eabe00
lGWu5tiDtWXeEELi2Q:883cdba395f259f826b408aab6485d32
NhRwCDyzRDyItJef4az:e7d14901665c5ecbb7a507fd17a9f58a
dDTZrYuWzCPffWXBpq0O:16a56aaebe1bc2090b213f36552c4bf7
8udHNNMTIMiLf2AxQUjfm:84749fa15a78891d2e50b770391413b1
YlsK8T0pKEo4IVIlxrT9Au:580579df111d078198458c2ba5ce120d
fsVC9GMeTAqbboV1EwwSr4E:37dcaec2386f65f0b2737d5266b1fbdf
Zg1hZ1Ik615z0Cb5lBlnDA9v:843bc188e1c76acaf3950694c43ce0b8
4vSMMNgCT8JIZhTW9PlZhmUtO:5bee6db5ff6b8764f1ce4e18d6845ccc
5HiwGDq1m3EZa6I8FhYN0KMFWC:726e1ef53e27a4941b009ac323997565
N8m7NiEVEapV2COwrPs0OY7DCYe:5c966552f3aaaab558765d5040fad934
fu0H8ZU5C3zoyzEks6ZOoE7eiWcM:ed8487a6d88b2f6d46f96c05430de5f7
p1QvmG2kvNfYHEd54IfmF41TzZYIL:38c9dbb586fd5e35202ad23c0566dc4b
0k1S0m4xfeSSkHqOKlJjrWOmOyMDmk:4def3d1f7b10635d01b15649232f65ab
fdv6VmiPJnUs64ZmJAyiJQ5n3JLFdYQ:54654b057dabd9ba85ed1c95915bc383
i1bfMoosXBcfy64LGuaP4QKiKkUgoIXw:916c9afe7f7a97890327af479b6da4fb
zYCcdQv0iy5Gv0i1liGfZgnzr8GFGt26i:d1c3cbe6d55836fb7f8a802df14846ea
u9laubiS7PdXX585BeVIrkZqRvWanTiISu:462ef5d552088ff200ff84f6aeef590f
IdvRWsx0gkVegmi1UAcKQT5DUim3Q5np8J7:8523a1f98987b58861bdbd15745fe962
U2uL8FwcLIlCtMF3toYW8791O4f43O1OGmpE:4db493d146a441ba700d81d49d0daa9d
RLHskTVDvqxPFlBDiBvWwARpfI4LVkgCVOV6x:0119c93a724019cebe70f613fd35bada
HATXXys80Whsi4OJL4OkYq59xyUbkRzRhjE5IX:6442bacce54df55ae51fa0b9068f6e59
3zKbRS5vsHq6RenNeKbYMlGc3OlxJUlCjVE0Ezm:9cb708d55e11d360cd65c41c2e3a3cc1
W7CTOH6sLHuAkG6mzKyXki8NsThioHuaEWTjuQB5:7aff4fe695bacbd3ac7e32b7dd4c5442
oVvyssLR39EdiDQAnXJBu4CZRl8b2A6hmsFEKhmDh:cfe86813dfc55eec18639a824fb90db2
RGpknQxbpZwkrl1Dj2wJYDWg9qRgClnjRDu5kS6AHs:057c62947bf6b5cb93da3dcb7ebbd8fe
KZDBtNtPnij9piq6oShC2Z6nVh5RHDahsDJMh3rvbAu:bebba7f7cc483c30e1bb3b82bbb2fac9
rJLoYtvqmlnz6uuO2YOagiJSqCaLEB6VaH9Y1vfdHsDE:654cc3c00f3e854cab014dc17bb8c1f3
NYiFMXFT6fCmICYc3UX4sWSjiYnPhQjUFsqhfW0b1sozV:38256f07c334a3a520139c46aee2ad94
cBOXpJfcspkgC0nj98OroWxaMz24NsPQUuE94JcmZnssdG:2754301effd83effbb4dfe07008c9194
CdEhvS3STFiMAW4gCOA6OvGRHw40O8whb0oxJhgsMo5daZj:533e6b3fd709db0256148debaac0b996
DEKAj570CtVs8TOfVFtjeApxdDsgEip7M0xWVuMoOyGMilS4:c0a4a68cef17a5337c763031f077d9e7
RcdVCsjGVBNpJ3nmTK8E5LSK9pnhB5bi8e4Awd7iFKyfDLrmm:fb3379a0608b9d601979fde035bc3443
qRi1zS0P68g29p0ejrBnoJTZ8YsKfFXvm5nMOdsKbyCbOsfYJG:c24c865ec9d9f6b182b55c9eddd0d003
bYgUNfJ6POCCaOynrdrJNt8guNzAB97CXdnBjWxYBZrCEQQWT8w:d29ab59b204e351c4116962c988cf3a7
xruDM73cz291Qdfhm2FbuutXZ9DLTC7gTsKvqEyPwyHdBNkOFQQ0:46c3eec7d853104020f9d0b64ab36bce
aaN0ahBTJz0tSAP84dOrClu3YFIulpkmpYcp5E9F4ZZMqEKkIpCaA:6bec22cf0298005f78bb2810b63cb791
X4ptCJFSUR7JUnFpmajb0VruwC3L54cSYrbr1H9Mo7l9kQpwRyxHkP:0f9f6118236553295ec9ed09059a256c
2HhVjcPmVEDXVvuV7JSjI20YoHmMfxr85zTeCzqn4UbQfvBd4knDddr:6bb4066d9c579837407ae0adb55878b4
rLE4Q2mOXLy1eYirT3tz8O1cJbpdetvw4ZrLRDqEfOwjDFBmyULxzDz8:07fd4083f6dcfa1dbc4ea78e50764bc3
EPcJ9x53nmE4QUzVzW43rvfQgRdFkDEPjGoi3kcgxGbdrBYRnTKFfQmly:e5c9be18a70794c4eb4bcd244395d21f
pRJTlo2RDaKNdRaKSe1k2JyLk4R0g2pvJYeVd7CG8dkbVlLEpDOiccTm6B:30c343229571bf659c226b11cce3e3db
dctty3hDPlBhRzl1A6dcBsFfAHhkUeLYh5hGZpaEABMi0XarUdtlvZrVwz6:2938928319dff92039cbbe7a358bb667
hDH6LCdhstiWTJy2Am2S75cjq1AcoZ8vtGs49vcrPkdzUCrlPj3Me5WvWmxa:5b1e097fcf9dc3614a258d6715ad72c9
bwwu2Oo1arjPCnemPwxvFr8KnU5a6tb8QxsIcHAdYKTrXYDDk1YPjXqwHlwDF:024c8e168f317fdef5dc5ced003c29c2
xClVUUYrl1gVKxIy1bJT0o2NFplQLPm9hyU1jJjuAqgbOPzFR8oHnhlSwwz8bM:87442425b29c1305f000656aee84b2a1
7iaS9jrjESzKTdpjJ6hXEv5PerbDqcfnkg6kpotT6TuPXJYwF6k9rfPwxQZNT51:d35bd741d86171020f840a0df5c24be7
4cXerbyb7ivN55BrBxBAPGfb3W1HFXyzZlEhn2jkaFX5Apn1NOsskyueklLQ8afY:6fdcd0b9e896a8c67d8f90a0f35a5ba3
lJ6yCfJDKxzbMM2prkICI2GTdizciF1ufXTH3sbDQAEsdxIuHgWg8t0bCpeLU560S:709cbe1db3f9719110b721ec70bcc934
PxL8zfO0K637EoFKbMIvCKXSOzDKvttaR59gaN6LK0JffeQgQpBjZp1EOufaOyavta:468616c4198bb4a36080b803db2f3711
BuNy5opEtuI9Bpp2zeirJXLOXqdYLwYdQBBLPRg9bP9C5ovutEL2rnHeDK2e7QiNiKp:8273d6ec6a2489f65a0f395e0b03088c
YrvYtaXV5bhquL1n3fU8Iv0NsHV6QwlFNHt6IgSEh0UBCMOvRztqVk4dSQaycm3PTmMs:6c1d3c5898098f9dae18706204d3ad65
tuWAlH2NkH827siSCdVkTVIV8CBRPe98zWzKtsnDZmvWEEE6HqhrcQdaiFSYJRX8EmJYE:72e2e001ee1245697c28d594720239c5
Xru9NgEiLAQ1I74ob4xGNlfu8eZlONQB5aAJh4RSvyK3vEixzF3cR9xQem1S0HjVIKuPEc:b93e271b162594b9ae05fc6d4a50e372
y0A8UWD2j3yd6fcuVhGNZxl9iTJDzeFX46WPSqH2k65gb8AXf7A5uL4DuE7UIDIDzuiiLQb:275af9815463faec6b2f2c127fe3c7fd
VM72O6sBcq17LD2fYwcI4Fgy5BHecogY0dQF997bpZ90s15gn8OiEVHzmeEosKcjOTOOSMP8:8bfa50075d7c3a882b9893e3cdb08cbf
BPYTGU0UTF3nqANCFhRXSU7wDMaloQkPw9z2TpNCUHQb7tDDBkqj4nGxZHJeoUUKTjCDzg6kN:749bd2616e988cb051e45ac008e6d39a
MlL5OewZvGTJchzKRDYBnykx1NW8J6LminhWsELNavmdDMNkgCMt0XRSBDQbAsnIFvvXZ7B0tO:ea05dde9ac4974a78fee0749b2ad48f1
4WqH7waKQ1xxJZ1q0sId1ezm2oktfD79qoHoLHYrymP8bHp2ZX6Q2v3UJeeOIbOZqld1S2jheYf:859f0e758645e508e5828e82127687fd
gwu8li52bqWV000Fz1joiur9dkikp7qvtKuFTpyUGlGwcw2BobPww7wziEJxCZSWAdsktQ402Bn4:894b9582ad2ba6dd21b7af0786edc094
YfvchbzD9V3hqDEStnFUrX4KEZAxqNreTNh0PGuOroWIRrrbFWVWKPxfEYCVCT0mx7ccDxRVMDttV:8789827dfa67cab879fe2450144bbc42
Kuqxfd8VBdppGaSp14NY7Psuwe60xqL8L2wR6uCxySN4TvkKpXymCRHZVEPjVriwkFegZGEophj8Ct:953f3108feda90b3e50cb84d49dfe52a
JSh74KPCzA6fKRxSeIoiOePjs4qB3TVCC3AxDfZ3GW8gDwZI4dQJhwSAq92kTNMlHcIaiy3Ok2VNpLm:a5eafa5935ea4e653eb06568717e9b3c
jP2SXpBnGBfQkTswqv5rDtliDe4htFtClml8BNm7fsNpc6MtrHKV0V4uZYBjuVMF7YEyBQwH99Wb5zun:d478a32858b06566e561b1d83f109deb
749XQ3iGRKPcvsIsgcRITdfSd2T9reno9mcQflmXV2ZgkxzrAgZjk42oXMneQAtQMwwSHJFsBuyMSY4i5:dc9dd71a5693cee18313d58e7b46ac94
TsfOktBXHGE7ZkKlGsSfXbHiAk1oe9jYspDDJ4qgBUer5OCC6lHUnf2NqU1vTbjcqNFZHW69HaqCOT4L4C:664d725bff24300028c04efa478586f5
vhIoVY8NjSOtU56qUEdREKRUdvD86yBsG9GrYF48nIs8DppoTs5o3N879B55ZwovwVNkqHiEgBCKQSZzbVN:9fefcf468244dc31094944a9b5da17d8
4yM2xdXt3kIpHt2RJA0dHrGi7o8xpTbkiNcGR00Lby1IRUqqlguSy01wfZUvJVGRzJohzeTBCK9kupAFwVoV:f7e49c70b456ccaef926944d65e69c04
MfhS41dDNKldjzlJD4a6FagZqHvMtJy5PFNKxRebspeBPqbjkbgPcmFsU05dAt9gZMQmuUoMatePJfZUh5Aki:6fafd59080dfb5ba033b4d9748e93733
5C2Wx6nQ5tFI9S3U6GUqKAZQqKYla8u5AnSYj5EfoaOnSI8PeT5PkVvAvjMvi6rIj9xt41zi1dwKLvqQemvp7Q:2da06196de327f2ef61faa7ed9861a11
Pt0sP9o6Iy65S0Wh9OlwpXSFDWSYbPFR9w9OvoL3NHYwyLDxqOUFCCb5pTTrzp8zL8d6nO01mPnKqQ7HvScXkeT:15aef23f6dfd9cbb48a4c01ea2639a67
AYDRoSPNtOQq2vhTHW7hdOfKaXNX8SHyGafUSVxcAesszzc7m9fpNk0O8DB7l9wR9CCRoZTO4c7tCarO9w3NHTrG:901154a48fa1728dceb683bb991ecd9b
nTDyS0gRsJzGziktliNNi4rirlWY5euiXYHGOOo7nODMXO68WJL5D3dUeZJ933iRSQoxuDuIhXl4Ch2pQEkkxn4Cd:fffda1646b278c463981ac8cc2bc9ecf
EC7yKPgq3ELwYjEMuyeCruiVEsjSnLVRdTfOym4sQGPFPjhaImCZHLLb4UTiwF0zp6dNi86ZFLvlVDltQOinqUokE8:97fcc3fb6167cee03edd291aee6d0112
t0EjqTguxptteY5A9Q3laJZiWg3brx9KoDUEn0YLfh5k60K5HDgHc6QZcJ0u7Z4ltPQHFEiLMeVI5wEC0La3HRSKrI4:5bf8230014271580f155ab66362640a9
oyZASEgqaLzMnDyWyUq1K22DzJma1U5fjUG2pNiqoH2CaqpzbGqMyifXSsYJdTPmEmfUZnbdU3GVkVkmrBYPK4Dsms2p:d6bb9aac1b769750dd69a765662d6862
bICGUHqJVrNFljrw4BIm3x2DrvWEOPTPnmm7UDHFVka6tstnU2ANpshHO4bsJV8W8kUSOrnzBowUGP8qIz4YRcvw6xOFi:44c85e04b41758acfb5d03635b79ccc1
NsgYdZCuc1WroHY3GfC59qNbL979ODhrWofZdIkgA7xOElIbra6qrJs3Jq2nU9FGoLwsjGyJDWo88W9pX5GeFY7ff0sZ0X:96532ad87cf4db5b717c18c59a582f11
weySwIoViSIwQGjPW7Lsbhh9mm0FcQsyURghqv3ydBVU74AT1bc3jj3vwT1yzjXk14shplGtMrdKmDueFw8OG1a2L1rbbem:9ebc6cca2dedd6a1887e9451600d9c75
38EboPHHsZK2benfJKeog6p9HgrJiwVcupdJ5KgnzRgBWtHw3LKaHZ9f6qPfMBh7RkHMVO0lw7MirkEkWflu5lz2Bf7eGfcn:90e09edc4d65a78013a8b05e385b8f85
qJ0bnQnJOZS5amgWsrgnCGg4Le9ijbvzLlBY2OyGEhBEtIqMZx0r36mFalNkm9JXlbMdPbKkicOCK5eAt5Rm14S2fvcsuMgFN:7302094112fb58685966d1ea85ab41f1
TJtUjD3miwXdBnxw5JnZrg4Nin8UtDxcdQXmjRzrdmvEzS0usejKkeosrmdLPBOShC5rjvJnH5R7OIr7MBIXFWgX8jzOKexSGt:2072a258feb891a85ad61ac49b7fc27d
aQOJ3mFKjjiAhV2PIyBPlAYLezPChmWh2ARWNmxWwGnDrgi0EKFQaurfT7I1ku9cVQZy3mlpSzTaF109CwPCR7HA5gsfArhl87J:33b6b476a04fc00c6c21f8e452b7fac8
2jVi2k12PTTPlgvNXslonrGsRYA3XxCPHnXzyOsexbTJiem5xyjLQP3yEurrSU6q8TPwy8AWakvioIdM7mnN2hbwBtNkdKAbugyS:69dbecd003bbb2e6beebb7c6f0dea396
eYEf90nnyq0wDd9wkbSMuwWx6myqtW9xLECKuQYT6OfASfW3gFFA2rYZEmgX9flLJOm4ubOrQURy0Er69Wx1elQJyWwy2I9CmmxGn:2072fb1f5bf2a29e4a47f30387504bff
cX3WErM9JJ8w60BhHaGux4m54PhgcNteAm15EE4dd3A9T1hr1NLoIYjCErIG42LveDqJ7lMaedaY4hgVVRauGkWaBvHvojQtM72JjF:9d0955f2a9ae2a6f198ec73ba07b3a89
KoIKdCSjoE1oYxyLyZ76llg2ESZws922nADq3lAhPrwEPUfdKdaVogNTZDghC8aQzD7sOxAuOWYuH4xh7y3lEH4taaBD9Bky4hRJFh3:52cc7b4497e4996eae47d9b9491b4248
k3SEAM2IJqBUV8Pe9QGCP7Ne2VVCq2wAVe5mQXUqemaZkQ4kwANcyqhrb2TrUfSGkOT1CEhGQiv1YqblQOoe5vv6nfo8l7FvLomdSuKz:839f754bb9e0737e382c9c3a50ef90dc
C6qrmsC27Q72ctYpycoJ9T5L7iPQCpf5lvmxOPqLvnDxGrN5u1FuLA5IJLpc0u7bQkzuZP5k3yIAQmvbe1vPsro12D4TX1VEbkY00UbUj:0368fd7ada845c7e729812abd4e2caca
Kk0XG11xnHPFVHyoBhct2Hudwl4gnOqXZgLv7CSl9yQV6eaxlcQeKbi6ndcASsoHyZ2wsLHsand6sdtDgaIQcRMpUOQC759wV2TdErwEEz:6720e62982537149457b1160a7666fdf
AWCUqIUZpXG1crGTTEOT0AMKDg1ZLvpbiSWzqHoGuVxwc4gWyUGpkjZOpQE1cT2lBPKSmZoGKM3NG9A4U7j4g9JwPenRYg2qVD97soD30wH:f9f31720b18c15761a27d5ea256475ce
xv8sf5CabBKxgOLYCRRTCkSzD62xWo5kK3CQY4Q0vrnC5YryF8h8jZxMWpaIE5TfYlVMgCCBTQ4PFlebjm9scw5YM5xh0gnPr9rxBUZlBU16:4c907861ff26cea69b076ce254dc4dce
557fh7ykuu87pvfgCsVTrdhTY74pRWlNRjTZhi9BC8zStEZVXLFeONYCLTRsF36ndPckYcLqaa8uFYgszMxdqlG15okAigNlWQvKT71U80fDO:8add8a908c2d9d3aeb7a5187593d528e
mVdYjhfvO6q3gQbnumabIKUGBFEsK76WT9L3hQoVNEOUlQ7G37HBIs7aXCsxzokjoVcvCrhfVWZ7CXDvUaXssVtgnLDNZOXez0A2hH83tY0WLu:94e829e57e015bca0664146edeb3bb85
iwu5PNR8U5JnJz2xDrn4kvBioV6fHIzP4ULJytHiphwYHolbGz5QUwZ8iVdQuDvon7YLrwTGDfvbtGcZ57GPuvOChSiBlEQyCE9T0TquYLvihx8:a3a304e2eb2df9f417ad8f74337f02a8
dvETPZxi7gryrVeQoJQ7AzoUoZgw7EvbZZUFPiNMo4bGQgm4P22fsg0G6633Bo4qdPWTXzvcuwIbCV5iOYn65eC1bvVCKPTNuFxi5SkpeTqHFmPj:4532acd586929f8014a1e1f21a79eb09
b3f67HX93IBEokhJPESKn30sMh0iuFrvzwsw4fwYOXs38ADN5loiefBRnrZHWg3mDlJxr5m5SEZQ5su0EJ9JPAr2RhzEosQREqfVlrR3WGKRZ4IuE:2463d67feb941be92ef4717570a8d3e8
H3j7kbPsBkGU1ooht9sLQmynWfcRNwmlqviBxYTY9qI0EW8YWrzCDYPp3S7HfjSvF1X2QHQPXpGrcFfZWFrpt7FnQCVVMEhhv49cCQSp6pH9UMYGig:1c68b2c469615defe1f6093bc789f352
VCdrPTUBFw5MErHEt9lcphrWgbz5HHbtaeKPYvhuRc7mtEQNE1PT86Gf85aFDcZDhAj5VqqDsnPMRvpmm56l2Cr0ysw1uluBMDxxUN0c0GPI15Ve0Rp:8b6daa0852966f3ccd6684929f9f79fa
SkHIIZ4AjgULSo8g8M7bCDRbvMWzCDPlOmTmbOMlVxWDMVKLxHNZau0w6NVzgBKVNu7P8K0U8NnKyYlVvYLwiBSfeEFl5f6Jz4oyFfiDSwdgko1GdC2m:29480d67d5cd2442fcde2b777d3139be
4Lr9gWlb2iLsdZRI5ZbOmfUxtM3wfWJay0aENmGFuhXx7E62u7GHcr4w4YT9Lsa9jaDWm9sHgf5dK165ZDC1UxoOl8OXqOWzPpm2pEzvKvzkmwgbZI3J6:3d8762767098adb53c9155af5219d731
hoig35wHRWn78fmDFId07oduuej38T0g78m04IyLvMJuRVXnt1erfhLAmUukEkqCjDtdcRPxtoRbaFptwtLCAm3N7nXCye4IHnMJ5r7pGOqGkF0GZB9qO2:f80bde8ec42cbe0d9ba9355759067d95
3MfReEV8dtmP2h7ZwDEN9OjZl8rko6ciIiZMNLLQ4XwW5tMrWg4WVdM7c3hrZkzytpb70NOVBaHwtkOgrI2cMF9Pzh6pswNMMOJDrnoSnWfH6TNnsFq4bqK:4841a8a446867b456f2689130bd948d0
AyGPQ3ssF718pfn8t9d05WDjf9AaaQKAfhfVayeGw6FLbSJuRNkNAOWFNnFO3ff9mlUnJYT5VoHW6ghOUrrk6ePKrlolqtkDE5QeUAaFPHrLOyqzQIJMMpm4:bb5539067c315ba71597d9f31547eed1
ABgRVrlpn2u7svD73UJIj070yGDbVQ5mhc33ufjH7DFpY8nSSXq2NnTbUncG38SbbM4vRd2OHx4wviee5u7JIRKs4NYYMH0NjU90Y1FwpASLIXQEiOdQwO9qr:98599c3b2679f7ed8945be7195b5e29a
Xf4u5HEQGvEy01OrJqZx7tFMxm8blpt9untpVX6sjAR9rwq1NgoKAUnX6lYhAigVwAbho7AxyixPEOHiUWSlG58DhXLI5Rtriuzxs0UQ8iwDW3LHQuSnpQQxEr:cfd3bb4e46207b0fd77ad7546f893308
5z9pRhKgE26pI4x4ykY9RFtASJhy6LP2aPiSWTYrLVGkQ4fef4dXzwnh6uG26mV714PONE5ppCz5xFaCAepZAC7wXDpTPbQH5wlI0gXfJnb6TczjgO9Hh745BjP:2897d17331120d58319ea062da1d4c76
hkwOgSZOI6BOGPQCyS2IYHHvOFpK6zpdKMRQuHu3DWH9CnC06uzV17hGCxhyWxBx9jdtQymjlUjN8LE463QY1XvtkCR7ZjuZsyi8Wvi8grLe3g90aPP1DakND2Kt:195cf9cac4bcdf963fed9d58164d2a3b
c4iusrtfMBdSTPXMVWDVCiN5iXIMQj5SeddwuwC7YGQHlDk6qNSSVvN4jmGZvCIzGLW18pYWVEt68D3ohL637KYgWv5iXEIufuldK90vEjrCNkQVWNOTnD09YViMq:d0d2202e17ea942f5be7c203ba872597
Q6wbhALhq6LzoedzUYlyD5L6VLVHTxXztjAAJbI08jpmntL8iXwLT8IEJtcsQ0SaajBKl9AjiQww98vhWRTFPrapLdIs3qs4ATELT4V2LioKgKS3sCy8UznvDVOwc6:cef584a91fbb5144fb0221cd7c8a69cc
qDQVeAQ0sr8H2fhL8zdxxYXLkqxYMzU33BP71vYkNWRF2Zh0oloM9bojrL74bSY4kD2cZQmCD38vSfm7qAKqBYASAyNChBwr4ptT5GmzzkUiq6gHxQXYFnH6LkySLVa:362f3a9f882ea1eb13b3143bd3d9b046
Gauq60CvqNgy3dFAAmnpRBWNc6cVIXWfXgvT6XfmAlLuohUPt84azQOCNQnmE9rrgMbcAhp1s1lH86mr4hstY6VCN9Oh96zfJBi9Ix10pcyn9LE3SXmGU89x7OF7K5cj:0010b57772b6f90111b35019d7e2b567
wkj4IaVXdjacUF5Dsiacg9AeN6bo1dxnoHiWI3JLcKOXfKqyTrBZq244Y5iQ9G4odcaL5Kn8k1VqCcOltfbK85FW1ND1jxfnAfYwQbu03ggvjVHD0IeYEJLwmfnwCsK3y:5112b72e7cc23d008187d78591d8f2a1
ypeAK5tRcP0XntOWxDq7eMkkZGNigGHFWL5wHpdJ54wjoa5MDlJIYTTOqwXx2u2O6YbDefda9zjoAe04qADfjmTzIGXB1QfXEhrJmuJmTSAkXreeRHj1T2rrzeSqUYep6v:26cf45d6bf39e0dca0a22ccf4c165f19
YjPyvygWJ4eN8VkhN4a4mz85QSUU8QfWZUkljr8SmdwkZHsCBswOSuJydujbboY199djqb2DeoN3W5wooS26cCvgWEiYT6PS6S2mUUQZ9tTVpf9EX1A0uW6hqf5acKS8sLl:097a2c7f593cf2d6c24500e9f061f7c5
cv11ulKgAPf4D7vtriqyyvtzx4iwmTHpeyhzJSFaxL50IqkZyAoXVImjCuFPed5iBcIbUebiP69nnjdMKrAv0NFt8b8ce3lF6THRXIZCFYQT8TwIbX42AAlyBjBFdMbavIRj:68af8b7d484df8d422cc3ae4457434f7
hHMNwsxucT3dGX57ngwPqXkuzmvVVc42JHG5Z3q2Nj5k71hk8D0praKQn5C88w0H4xDTrTL43Hf0znbyQ1O71oOfkgdiD4PxrjhI3TDWrIMRVNfCETAw8fBilEqPy6dQfko94:965d8a1d10ba4ce8bce71c95db40afd9
RVmqx3cb9EFT42Rau0lZg1od44jeii903lhrfjseNYXIQFIbwU0CLeGF8PKhXK8RVg81pq53eTCVpbWLVNexRK3QpEYdeXU03SRi9NcexEZMFMorqsP83IOicDmhq77jPPsPsu:21a8c0096eb3097d3089cdd98d848ad7
TPYJsumGMCYrAR0e03IdaYa78QNqv5KfLzO3T0zvsoNS6DXXwv0wk1uiH7z3398EzNyiDXEWchE8KsW6OMtZDO7bLw4Fw39VGx4alyWoGrmgK9doLwdfbbhMyciU5hGCFACRYo5:0e16faa149d47eae07089500c8127a15
uGiBghFuT2yZ3A6GY8OJ4WfxsFZ9ue5PKegRmLc6EAVx1S3PQHoLtu8LZYLk2GZDL6kXImTcNEzEmsk30yFjSEVIswSl3IOEEZrdbbfPGFjTYtNOSiXBMJa55SgYrVtVKKZMM4si:da58ef1e21abfca7d7de64b1127e54f0
AB1oVF2ENQ5qq6mlOsa6etSO4HrGCJP2bHhXcarPRn67jiiYBjUFCCkx9B3BbIucgBZiChXjuUgD2prtybZ1E9oDBh5CQqEWSu4kBStVDzpvPHPdIF4dEiG5qCx7T2TBnOLPx4B0u:b6c1df0903449eee559fd2e09821e921
Qm9n1dWw8Z1gwWwZktIdd9qSK6fFx50dh0q94dw22o8pavelPMoTLFBmBGSZCJ2KzjJtm6lfkkulGyxlbM4MhvZJ2Iyuhr5HAE1NBdTMndXTCk5DX0g4m6EeE3zMlujMZaprd83rcR:a8524b421b925093db7ff790e68ef551
bF269PWgK9cfeH9EjkZsWOsc6wb0NdHOJzKIfxOPwRUBoT5x4UPQz8SvuTl7XTLwim5oKJ37qOyFyt2sdI8DHRY1AaZoTBL2OGgogalxOK2c3Uu7ttA0apSAqHP99q2NW8229dpNOiQ:bee21f98a7f556fc8f8306e8150f3fcd
I3aGmt6dtmVUCtzCsPu5CtXvs1L6GTMeMNKZ7H2rUNbm6BPph9kKt85L0GIxqkB3YcSVJKctoeGlGlANkLnEKjfBQOY6Yp0MrIx1jAlyERJa29OdL2HlcNMSrAPgQFT8ehZnHbMcSldL:0057abb62cb10930c83e6630c5c83369
kSOWKm7N0JwhaboQH7PMeF0LGCOoNRZYzEKaRHNHh9Phb3YJ0El49bGGDkUg2K5sfFsWd6tk68s82hHSLSNLTkhnv1unCpGIUYuXVO8RNqPFyno952KPdR2yJmMcCiKm74aSJ9AmqgROt:969bb630abe8a0ec74aae4c8b67302aa
6NoYo425XqFa3HCbhP9cQSLPiX5pFKICnXrCRjyEA3FtB7vJMuMscn7lb3AHD9aRWrjEAH90BEj3BFDeZfw2tuevnE2QD3yqkIUKgUBHoLBQg4U69gYsB2OOxHv1AThKr2lxNM52oxIvrt:56a9dc86fc1a095cc6ef4efb6ae9afad
rrAgJ19oGxV1olLwV3p6q2JmVYJuj1NBieHS5H6B4Rti24FOXVLeNkqz90TiRwJ0AhIwOEXIlgRebmSY7t2LEjaEaTMSgmIHtg37LRGXXn2ZAKNy4F9zPa3PUGy1S6ybds9P9FCXTuMj5pH:3ff407fcf0c3281f13e998d35bc6deda
ZVHpBIihsOFkxCJzG1yl04NON8qW3GtTs18Uzh1SWx2jZBIvs7Gj1kYEiercLLV4CUO11FJO2CYS3xemuLwmWkR4p871JSVcCz3u5D9Y5XGYkLbEmxR8Iy3Yw0P6JA8c02wVvwJrjggtRhY4:9d1b6abbf140601314fe1d13216879d8
FF2e4W2rMHnl8mnYfKKB6jSgqYz86o2BUVGOHy5k5jw3wKSBks3hCLnSzNQv2JXMvtr2SndNwAHTaqllIesaQGTfjABb9pOuJ5nrir5FRDoRTA2sFlslRcr0CS2C8GXIBbZKsVfaouR8VKqqV:f971ea3718e9a46f0af2c6012a0cf667
JBCLSteBmGzSt7kuXVNI0NXlvV65cmRLOknwErXQXnzhkJB8ueGlSuxdgu8iGQTk06HuouabHziS9KQuPnGxI3AOnzWUfG56DDqRXrTvR1eRBVbg9IEHC4mQu8BAFww80NQO4z9MqetS0vZZ4t:2c5a8aa2167323eb763f22cfccccad23
xwoJcIID8d0ucQ7Su2huExzYgpk5fepCADcCbK6aOXvQDsyYlFjQ3IE9YO53SkvjOxMQ8IQMvbsZuROFmXmgwQgkvbdemzn0XZGVynyUoQKJxoeKcrQz7WJtYNxbcL10ByMZLaJ0RkzfyEPBVw0:80096a44b520f49518f6488dcc0f6fe3
TjAc8dA9gc17DqJs1J2SrchwBMc3xzTgIjjHnKHuMyrfP0yHArpRuxeV9gOwFyDeINLWnjg0Ix5nouUPMaw6xASwHxScVmhuZTgc2m2AKYOZjyEVy1SVsAiZX12Jda32T95MmYcXNRMWfhIE9qqr:cea23638d93e7c681487e7007dc259c1
QyhEpaesa7lU7hwk6z7JgKwlR5QRl682ugxKgB3hye1wmnGiMESSfe4X9LElHCe2JBCP2vXrAOOM1kVEPDnUIhHIScTqEXiepLUh6HzxldadoVH3pUOX2mvLof1T3aXiLHpIeP5qS5t6R0Z6KEUDQ:a0c03a4c94b06f1f586dceac647ca1b0
ge4m5Nf5BnH8Dfdiat06NWNDMKegvHXCMROHvTDW7aVBfZJgiAcVnQp0qu7M2UeECTbYCFLAFw1LmBRu1TgfAw6RR3tJOydghf5KKGag31RfszAkjGqJ3mqKgJjU8n1ftWP4sPalHSqZiQarnr0hDh:11f55d65b291b5df4d260fad9c13bfc1
RJQ1uOflUy7zsqftBMcSCkyCCMUsTnABXgsiVytGWr5fHbz8NBRfMfIeSsxBGX2u4lCQJWmwdiBLj0KWsr24xA8fTvGpjyKdKdTj05F3d7FmXf9gw1aTs8YbuvrDU1Hu5qDVm9Pq6kDUqC0Mu0wMZlN:90ba49ab38fa3aca00e32bd9e6e391f7
kG5OqXlLSCeEOej4auxVK0LnkCJWsVADrviIitj1WovAtFvtZSfAIRNSknFC956rreZAy9rkoNKHj619O6znNd5XrAqqwnHNrxdQxv0M9Bjixbhc7HpLKlz1LQi83PMldP1BbSdajmjHoqJlXYXy9nqL:de552f005aa3ed3ca42764f2b0bdbe52
3yKXem9i20J3JM4SZdqdu0phPdGPqWrkk18pd7x6X6ZwJUfyXwChmSp2VVHcIYnSPl8TivP6sFt1pzqcVTk8Bz0nlxqTlDC1OAKWWqTeVc6bBmdn5omxOcLZA1JMubE8CfVovEthHpi8CmmyAyVfBw42n:35d31a65e026c0a841444bf05913a66a
EEHG8G8drnzWGGtWOso1FMr3yxvUsn8WSFsQcrUuFjgcQAYvSdmoPEhd1DYjQW6yspfuG0PcaVfQm3b57njNSrQJUFTBrP0K455BWKDWwICICEDASNdA4UaOpTgHy6is2dUOOnAaW3JpxdPgQSGKCGpSqv:e4d837a3e2e097d6468e8c0cae0a3dc4
pOrx7kBRYf5pgSjP8HTNNzwkaZQpH6KWL2kJdLr2QniW5rC3ZlHCK3MKSt0pqBc2umBxXSpE5xq1PTVE4s7FmKg546lkxncSAEfomvSiT99ySU3Nd0ipAolvlwFIKIqadwpqR7yA6yZPjSsmIAC9PNu0a0z:b9950857cf38415a4f8ffea083b35a35
KzQLCc0SUYhkVF9AOST1qjuqZ8LP8CfJ1VltXbcI0jSLPSbuAUlR4G8UFJAEbFd3qoneqpMgzvSeeTyEEKmzhkjMTKgVfjOwy1BPheWGAELEo99SKliRvste2z07JEu8wVNE0zaq4MUiLT1m493psmukMkil:18d0f91fc5774e534dd00db638892d38
PDjcoXGfnHvhjfpVZh7Ug09zduJPFS1klamAYTFbq0iz6ykWFhGM8Gbb0LQvtIGFIS5wBBySsGhp4sbAAHcyddA4Ohp8PVDoEyL609OiFWyAfz0F734b6u5LClJh6cvBB6ys5dABZZ25oTAmMvnJQikitUpqW:ffe496aa4026e025e192ce743ddc3353
KSoHgGDj749WW4bFvniRARxJ1F3MUkyv2N39kwjiri4dd6JzkRgLzDkr9e3Uosgh59hgwqoNyjRCfq1p880xCbPBgIlu0sCWrJ2OZhroq8QwpIMoGDM9ErAKZM5Q5yDm7w1XDibUhSgwrSKYmmXREost1naWME:1772be1fd97d868fa906fc7ecbd631fa
9K00xu8zefilCz4cngpbXTpfdgte35I2FyS2SRsXWA8p02re8HfWqv1uCkyvg6nLv5EeNX1znZOdS6iREnDUJvfbGDwMAJnVFSZiFRITGnXpk56OjAzSVETr8g3yQhkl09uvQ3fnh2CrXz66ZwPL1y2ZEVok3yw:716ff4a7e0b94c345c064e8026184f1b
TxRfeLkrSdUa1k6Y9tDKk2D9wpnHiM429L8emiv5mg5dq21qlu1vmuvJJJhSlbKlNIp0RLV3RR68JYoVifgvzC48cbQndrzQ0OGHqsB79y5ImktvpAQO3KN5MujPLIwBn39DlBBkZxTcHdxWDdBxOesrzB7batCn:2b1512152159176c840992858667086d
mCRy4iJTGsWevj0Znsm1xFjW6gX6zqkMT1bNaKxH2jLotCdGUqyhVH4SNSPd8ZQSRRvSrTpk20ylsCSdTglENfmrX2vWRbEyTZhaJwlBnJNFcvJVCUqgZDxNvTzcU3BETIEs5QUiqxOsTnel7urXXPAjy0ljTNNDm:c1a42202c9e340dc32b3d1dad52e9994
iWhYGqe44xOiBZgVh3JWuSkk4u4I7xUfk24Ri8LdwpmXfsJmli8G1jQWDKuA8eGi7AZgzBjVQvJWOi80r7wjgc5KNpaLtHTr8JxyaGkR2TDH2CxjzTtGVogzDhaXOUeMtBat8lb14FyW7V6xFq3rE9Q8gQVVA0xUsy:88f01ddb90e5cdf44ba572ba0e76d2d9
erJfiEKHrIso5YEZf8YV65BSQ2IeM1DQidWqHxYZ5hd15IQaHFVDBnmifVnSMQyVUkcrI0hD7kv2Tc3qIOU926i7SvPvbegWpidX8lr6wNZfPSvnxfnpcvnUQ3fS8lEnuIbtUtpg6elMXGZlMdAOyOzfHEYFQtTa2Vu:2a9be768680ad045a348f295a9b41c75
MeJSbO4NBADNdGdS5ShbpMZVsjFtbrNNvnwx1qatRE6UaaC5SJ68lWUE5pY6GCK2P6qHmq14UXOVYhRHRNF3ApxvFmscO34u0k1dBS7mGL7uSO2zsxsTNQfi2ykHrf1hqSkRBh47T2sBHk10IkJl1OuTcErtKiB01LIs:5112291da50cf9e7089cbfde73287fa7
TCqCuI42TWSr6rDX6XH8sYs26TTXvrgfTwHe5C6OoP6k6A82nF1Gujyq3iepzlujH1oCtlhH0eSWEQPSmGoGQNXJV2YldiuKaIdtTk1KoKwTrcBDIQaotXYfQMqTUKuVjxe3I6DXG0GX28rBOrQ8fEdVhuF25ZNemS8UO:9fad7d476b99fb8bf0b05fb6176c83e2
BIlse9ldrNRJtQOYUA6e5Y9UBe7kcSZDqbWva7ysKg14WGSG6OL1DKM5OJfRs5lI67474spFzhzlNiSJXtBq4elIOrq6mBFiyzgtSvYhCnsgwbPjuhzolLX02dWfFso3RunzQbHjp0pL15UmckAoVoeOs0TXi8R0t4q96X:91450aaa6821363e9f69c830a07992e5
jlNyXEtHRwRhKDwOhPF1COZkFieVibJrmmPaR9IyvqG6T3K1IgSbVHlqQqcYrLgEYVFGVd4gtBccvN43UM5FkgW0xYPPzWknHPTtSOAbgDeBg8v1LqwVxiMU6szweJKMpu579v9fZeH5d2WOjjAGscqpvQLzqmcFG8DFuCL:5f1727a5ac0bf6f60eaca9d3aadd0245
jHjeKc1olbORtR8IcPkC5wEB2C4yWxa64jbEm2Tx4xenfdVhS5KNsefkH9JuwKqqTrV5kEte2yr7Bdfk9PXrU3Cs2cMpM2GwkrsuWLyO9QMBUSLTyzbjsEBlGdBi67EhpXBby0QyHt0rbCaA2cJkGawceXlaVQhaDJc2ASr7:8af7562e8ea7d4ad26270fbedc0f6fa6
bhodKpDCrdNXd9ZhXbiIRqIkZKcptumuBBybQ2E75hU9gKg3LpBtFaDvKGK30XosoNu4FY2B5NAcnG5ZVxirxMM7jn1aafspSNkoBcPxPgz3NvSyS1QgEseNF5OGb6VUK58b7OzMUYGxjoW1gD8LVcoq73W9ZITADSBBx0ehP:e7965290358dad972ddb74d5f9d29c86
KFYZsQ5VOGHQVYOOKNEjx4MfweCniK2SfRHyyDkdaRUWFyAgc5pzZ2Em67zeHsXNaubyYmBY3lKzKbFM6UbWMG9JDzNaSAOSVQgK2IzVTakubQ68A8VdFUN8kr92SOKDuRempD89DjtFZqEqppuUa7TkoTc6xNARuE3K81JCb3:fac97c53f309793daf996840f7ce70bb
80jCgzSAjSy23ML5J9J9Q3EKDCCd5D63tgwAFe0PXpHQ1jMAjlAZe4zIx1LsvImOOIejN5YAkwgcGSDP33F8Y4GlWiEhQQWup1E2Xts7QyanhE2bxy9ltGxfP1nvIaQX1lPOF7MlwNzDhsEFQER9befQ5scDsSrj46Yz4ALqdaU:8daddbee29ee58f0c69cd22c508a4a19
kspP9Uw8VBdCxvE1OniYheMH9dipgjarCq7BauzVWDnjZSaD5isdn4Kn8TCf3DwvTuXTPnFBQTLFBL8xUrA8lblkUOpOiM917XLMaheQ0PlsqkPbBg9NhlX2ZdG8PGZMtBptIt9z8lSowyqYFqBMBpFrsbp8IfK1GZlfilFgwovT:fb4073dc97219ca86ed7dd6661f50fb8
MLHh18UtoqKHr0F05gRC62IfedmArHk4jRcbQXv4d5CvW7lRe3kaWSf0WrqeZA89ibbYYwT1slmejyWnr6ndOs3AKkEzLDzTEAItWrkeDxjMW5Zd2chGva76lCwW5VGAlfT8w4dZsmCeirhauoHQpEMA69n2UUs6Zc5l69bomEtuW:e0bb3c7980d2c2ebb239d51a71aec189
BvgP2X5xAwuzKmkuFhjHdvDcxSo1bINC4UiWIek8AEyaRJumQETUZnWn5ae7JSAEDIqbMBad5yeM8zYP3IzTVl6RmbPWUfqnOHpr8pu3OzGMYvrS3RBP2yxfAc1kiiyWPOdO3yIIXekLzCttjV9ckwhKy9VHhktXYwBSU9qIeBjD4D:27cf18ecd90d06868769a07beb2bcd9b
XdpWgzio9HnUev4xjTUKBFKRdOjnidR5g6Snv0BvyOfDaa1jTLUlhv2l9byhefdlcVyxMZSbEYEFZvPJ7zUfUNqUPO2TU55WRujuk2vOQ0kGmZfjpZo9CFUhjM13IXQph0KrS66zWg595bik0xkt24BcHt6fgMvnD5Fm2CLOIHNEIWP:059b48f18fd7d7723aae2c3795e5347a
zk0SdVkgsDcyJPTXjPsvHV7mtE0YdNDCYul2fv8yZbWyRFl0lElSqi5JNVyQz2jnmvfrRePGfC4W7gNiLZ117WBLIZs8RCl3XrlEv1bBt5orc1AOQsGOe7qNXJLEbXyZfKtBCv2Wqgdt8E7PWDu1BKFokgTl4ha9SuAkPt76AbpyFner:aa3c79c65347ab7fc8d1025470e0d9c2
Ryji9OxjVgvPovP6QghvzesZfIoLVs3DRcMRR91CgmhuIXro3yKsD2hJAwlmOePwhsdYr5qyhySQla4ezEw2xECX0NaFRQ1Y85Nq03O8rxOCxIGWd3PBxhpnUpTC5Kr4F4kwYZufm8HJHevL7advisJ3IswE2NyyIJVwyfBKe9kMeFobG:c4cf743ba233049d7d67939d4cd62849
sxPK6Ijo4NgIcFgMqDaw5LB5Ve9rsbYKyDlubuI68PEblLNCeNYapp5at4sM6gnuJzPLToHS4cTfOxHSbwTqLOr5JJIFQW0plF14UzNOBxUfUrYVNHbpmslV2TrIFhYQNPLyepcGMWLweAhR8j7kCi6v2n3yvSf9xRyC6A8JnK5rad98n7:fe2f0d0af8715856e515ad31484c9c7b
iPgfa8t4xOMCO9kcCgDAPRaL9aP9926hHcnIbGCzke2Zec2Gjv6Zc7Ac7gb687eGarecYGB9LtYPwQlFcivffVhd2ja0gewhwBjl7LkI58orPzXSIiYN46QWfQMmV9tiADtyeD79ClriLe0jmOWgKD3QkG26Fle6OyuT2rSuCaDeetnq8ax:afdbc28666c81961a7502645b83acd26
JDqpNWhKstOo8dJR51AqtATxEdLVcLjLfA1Tn9tFs8Uqbu87lyoFY83tbEeefnQkOH31Hnw0lhhmBfkNDyiswbLyGQDM4jWIRQzo4Wef4lrvrC9U0hdnjOMPvfrpoeX5VnkP9oU3AcpREoCvwFIFkkkFpCUEGIzsVK7UYSOpU37pizKE5jaf:c40fee7833bb9895d1c8cbde82808c0f
tkKJNvdk3DCOdzz1idh27eigESKz1KFlVfUzBXJvqb9tBzlJDtCByKH2tiskSXwE2g3u4DPlEPE6eQFI989yIHq1PIcyvy2oFWIzpoLU3gQ8WlHWjHkSeBKT9MhvbaKHWj7cxIWrOCqBOXnYuyHy0hh0UplWpWtc5qesY1KCu039NhX7GvwwC:d64d23dd7092cde6294689b8d85ad928
EnnTy99ksbgJgyx890sZUrC2fAzAXW3r9gbygaIzcO9jdwhdnA28Rv0XWqnJch1bo2zu38U6NTfRgnUuNMsv7jiTzwtBDkDRd2c60X3DG9lNm570HqmFzupZQIqjTT1WM4TDRM7oLi1YdYPKe1gElvt24KbNt3z6XizFVwTwFKlIz0jDRp83Lr:0040dad37e4a4f2f7e31f8c72e196669
VF1XilQIhEQH9B3SYys97R0glHwjA7BlCs9KDPiLkZijrc1gBkgz26FdE1n4ZOgr6f2AVbc51lfjn6pPhve9sKdWCqRr4YJ14CsPDuLvGQETNTzUfEUxeXjGd0Y8Ox0I0iotDZO9GjTj2j47NOvRBEoFvdEaBuJrDXL7Nq6jzQtrZnpD1LltfzZ:1d0493ccbb07bba2020f95c333d09b4d
BDtB4NbmhYXeCdbLCS5UHiaaaLmuRLjifDKahLmoA9s2duEGczqJIqKJ2X3JydREGsEN3Q1uQkmTEQqGgGgOXRnPEhp3l6xRp2vjInDpyP92wpzCWGgJoupTCENNBbvQ3QZC4tRs8Qlu6KXSh4swoHfQbStDTPkNvafptWHrDSMztAiBuAXI72p9:ed2ca6b040cdc6bbf309d1c3b1732f33
LJCvoN8KNeZ61xoug1Uzr21LsOjqHzqi8TEnwDX9HN5yajTgaEGrxx2Qcc6JCnSB6mOsQCrnfmMfwwmx0TPogI4jK1SdeBEaNisuKJHQWkWjG8GxSlLZTG8uxQxCh1CVaVgLvOriYdrvcXTV9vKT1IdppBSxsljt6p4s3mARqRdtFWeEhOo9nrpCS:48950b1e242269c1d979c32f280d3d07
7Zbj8ufnpygBYYSG1bbOfGWUUW7cbzU9pWinhoBwMIYAwG6oH83NFPxpBurDTcD8ZLm60NtClh3IN0XlZQYuvlJXF1ro4UnUwzQmd9PohIXVzK6oqUIM6izBaQQ4B4OXtv9wvOLCnynN9k1p5A11JrtJ8aEz4jmoFmL01mtfLHSKRK0NbRFK9Zk7ZO:1c8c3a388b815e54dc8bf2e4f31123ec
wUYJ8tWJjNWDSyaB8ScYvdG1OFQYDFC0przoKlXTZKmI8na75cVrgsiUXYIqtbgJsGX3SLMHlZgkcgh8i3polx99mIqGKxf233WMEykQoq0rw8pF1FTd3TcfrsL1PRUJKGle5vUjLLB8JRDBmnEg6GlnZXeEEYeeupjpL4znG0lgHOHU1c0XIcby0f3:535cfe2e6c7513da39fd20933ff52059
u47zonJEYD4b9Umfs1NdTP4Ho7IMXOFiIDIXQhrEKlGK6TQpKtsu8n1mkAZ7evgMYOzFVR9w2G7ZpNe06xk5KciUC7RGtYihCIMopMLhiI7ylboiyzd8BlT3jBzCpIKRgn6vZHD8fAxrCLz0bDZCOIwX9Vqou06Bd2W3Aq2GRpXj0na1QZuvyQjxCzMX:a86d69eed45319321f03b6ef4290abc0
qJotCbn2rfy8umhlAhdg7xCveLTGbwtr5HLxIYq03PZo16zreCycZ1x3ChAEDTVzrxW0ldRfSGtKCS1HlqKai84LfvfJf18woVnAY4GGA0g3J8BUpbVy9Q9ebeOg6MDkyQLmLiTc805HZxrfycDxSCCUHh0DTtOiaqvMyfOw6K3V7l1vovT6YmQvtHZd1:9340e257d273895e3e65095d0a555f1e
El2UHEiMjEITyuQJlcXHWUv9BRs8HwZbIRVfm4SvyqeXKVwWYkuL4PKwx3u4qk6Y2R4fMMAa3FYErkrgvLRqrsMOm7JCiFrbnmg0YH0SdOmuZDBkfjAwBdbNaLgihyjvKqlzXch0RulHYMS3Viqnms0m37vbvEw5UHvIKCyrXJYLmGE7PVk1dbe78z8u4u:5ff66931a6917063296926b6f4d7d9e1
qO2LnDeLU1lJCxg7vVSFNWGRTFhS0bdgQ6R3zWFkNQTfeZdzLV4pIB7s7eaXfe4VaMOzyjJb0th4jktV6oaEQ7WNcXAh1v327IrvS1wIkDDuOWfKbfeRd1ufPVnHggzeOQzwIW4Tqyd4liEmoJ4rBoxgaKOhQdlvULRsxMbOafIlncyBMsTdHgkHQYPx11S:43c30a670fb73282a315bdf288d13ef1
MCA40c6EccdopfNRS5AVCH5aoUPLMHuoaUj0WgvZiyexERfmNG8fd3gsOV3qtoEt9OjWUFL23QqyywKb2Ih5Cnng9hxsw2LvGVhrrTkkzBIYXjZPS6LkjYrj6PBsHdOeYVvfEFq46ZSU9IzRFa2PZjYVZpewsSBhDWn8sE3ptL9skyaPz2uocia1xeoQXPYr:f6f75178e489c99dc8dfb35b5af9fcef
BbpUFi9ZU9i4yiKXb4cddc4Bgih3X5kp7A9DJ9su9BpxKZlLTnOWpJnwRvpFrAVp1USAUbUTD9hdZsOIGtvV2JhK5xfnY0CPLlgwn1gQ1nTQGypdRLYJk6jguptjgWZR75dkWk0NxKu48J7PkWovTyBeNVnTHcBF8FQVPHzdhT8gs5VCSaXBzoFcaTWIVodUT:e17b7180ed052f6e36c4b6ed5efeb614
UFyr4CzNAG6wsyetlFSAbVBUH9GWdrFXW4fRwFvXbrjuQnN2TvsUhUFPTcBXt6Lg0QXnmiboAkIgyl8hHr2OcytVAUIURj0I0NWdWXrn7ZuwbtDJLwnNUHyks749hUI8yublhtyfjTBlc5UNr8qbFPmYNgXV1vTqQVBYe03oJvzLqjoIhFJNkVBX2pIULsardB:7cce2b04c2948b0e17b5d07ff62738c6
fis9x1uWCL6RjemTRwFjkxyShabireJxn2xKTSxmtt4DxgnfDSoOgNxoNyw5D6tRYQsIyPUS9OmxVzCpiR3puqE8P13iXmZM3hksW5aWKnjvNMKVtEbNVFLAwFIkSxXLF73s34eDhoZU0zGj4IXPezfA4OLMcyoHwi0qd4TliI598Ci2b5IfuOFpsh2vGh22pSi:02f3190d7394ee527f265686e3c7924d
tN2EWAAVz23s3Z09kFEzXMsiizbaPUj8xbCkC251VZjPPaOZFjptVHC36D4LoeJMfcWI5SzQIJwnJacekrx5Z0ZVtUxIY7k496C4P2LnB7Ab8DfjUDfKt4vNPTlDQvyQs1L7TnllkLmjessZVxzfsVS7E4Blq01IRDGBQRM1tZayhtn3RN8azR73Mzf2q7B8A7zg:a71d3e57f0ce7ac14fd66ed84660910a
Pc7918x9sL29o09NI7HkxMcNJDL9Bbghddg5cE4ufWttNt7lqEwOgzrQ23Qu5WC80J42dZmiLFBpZzBf3XTambQf5wA0j2Z9BU1FJdNkJfAyE1ExOoy1qe6lBwcKp2JRMBmlEZwe5WCzN6XrkljAqfMRCOsRG2zitLuYAQ2vCvVgrIyCTH39MGRfl9XR2m0lXlao1:686929f83255c337749af7a8c8c5c2f9
2KunvAF4Y8Ov2O8yvdJv0rnciz0tzeksOFGa5b4TaJf2ndrJh158iibqI1J75UqKpWKlYF5YeaRse91m0Wk85lpDcZB8JRIZEjasP5gT6YcbX3nNPxMLJ1fMRGKrojgSthb8diSa6UbUNoytMa5l2aYJGy0VI7E1fF0iNIiKtju7ySqaTlmLmalTzcE899ZfFQnjpw:1730a97c48a011f107d3e25bba89ccb0
UIGeF4XW5G8hikhu4HwyFwyvB4bPdrbXZH2vCPixmhEvsMQmjcLPz9a04cG7tHVjfNFItN5GVA2dcIpmL12a121V5yTo5Eybh3JBHF7s50w8yMk9Dcav52g0qZfmtEoByY35t1ooRLmfnxfRAgcv9imq8sCrWQTkFMgYDuck6pqjMw0cD2yCbLS93lrQ2b0xN7lhsns:3c7be948aa552fbc393163542d8d5f2a
oDJxf5oizhH1jiJicUATNBKafHlnZzFe2fB8bQhAYOr7X1g0MGJp7jpdRBqG0WKS2cR3TYtHCLEqDKgfhQufZKiGbzd1lOKdRr7Agr7I2CZvd6BkMVpCwy9yXcq9R0dyiaYorWXkpNFsJ6ClSSNfgMD3PT3wK6VT6K8owWyMz4481xjKfXPmzjfe29BD5mm2Wlhj7FVw:fc2383c53eff18b09d6bc147581225cd
AQusdE2jrIvQRBVKAndwzqomCvwz1iVrZgJ2KCc2ayJRZusqIwM8M0kemHEdPqvEw5x7xA0yYzfOTy4sUHqxxACK86NOm9jJ4HHs7xQWWWAGkvY42erqFT0DQEi2DrCyY9QVwwIii9PsuDnnIFEdovH50PXt6pSVpyHLUfU3ezmz3zMB5hFkCcfsS3MOiuzyT79EcUxgj:a0a16dffbf2c949c4c02912e33d27cfd
KFdaiE5p9fSckbUdNzm8ZKRWKmYuJ5EjBIkJcfZclIevJZznoMmdn4ZYgNiQJN0alaKofzqq7vMHlcUzP6C30sR760NFDDFPDf3JFk0DGDaRFVhlRKeI2VFYLjufM0Ug6OPBYF5v85cEQjQyTU7MGDBiMVnoM8uIM9aBF5WE1Y9I8P6SA3v6wWe9IsnkqIT3ITudPgIH5H:7fed151b4c237d79e78ae9bac4e71522
f4nbNN58JsUOrs7FDndmhL5BPLHniDVnx8paLl8lET0WbXrEbvRi6NKMoiZxLLK9J0alc9xGSnsUbKpd56l1JVDX4tkF5VFFMFQOEeknsD7tdmw8iI0SuuFoOQTJCoefUVTpZ4ChyABCMXBVvrDPM93rQN0j25oMR81G3uOrUfTx3kipCM5fMZwtCnCvjR80P0xJkc04iKs:5b461b2d813cfdf5a8d4c8a2dd194eac
bUBrndmsQbP9E2C0bkY20OpzYrpTzZcBKDSYG5gn6VnBOPrPZfHPUXfJfusFkv6VYPJvKPIHBV8fBqVrvt6fgbOlw7QHsNshs1C3HaA9VyenOZEaiBgpDVBZThxbVPidHK6fLxexWjKB9eBiGIx9tYYc6meR1nVzYREzfIX1SxsRCUZ8sm7LbWO78SO06zpUhUkms7eavwSY:907931486799962863d31a160722dda2
gHXzTUbVHPSGyxwuWMf4wphPnlQJIyxP5keP5fADVi9jGwEt8JnvZvadGRMffaUakYQg3gJPpJYV5sf42spROqUl7wqcxadHYTOS0nxgWm2SE7Mxq1e4iZffmviJwlgl5V4W8s2VEUD9Sfw87A3fp9lButaRFh2A3Wm1efNI0gIIm4HkvAqLzBdUVeBqmtRfg47k9KTZRsx4m:e6d3d2685686be574f4c05c8fc478dfc
4eIvvkUWnEIr6YEqFJwzGHzhroYJLTWGXv1iFM5ShDadsEuYdQoKoNRF1GfDp2adn2mSEiBLMBP469TaQ7K4KsACZG6ey6iLYvutEV4gnKbkKUkrRUlsdMU2iR7GNfiCBCV5oQcCrdM1XWiEHuXK7ICfpzM3FVv7oh3CY55f8H7Vugq1AeCyM5D1ugV01gYfoRIcMEhKcfwwlN:8d5466a49d6c2764f66471459cf9404f
oMRRauMEl7K7XCeLIsshu5LCGNI29TGovo5vIHZUEARs2V4AdnHyij1OWzHWidaEs6Z0EPK9frs7dmygz6FIfxn1WUO5YOzgLp6f4Hf9Zx62Kv9arEIx1VpOG3JuI9Bjoypi5vhUSeMsAMtRg1f8NuMjnmEVl55ztlIpGP9oUMHkp12vS7uwr7GFjarv6nUpysE48EjShR2wIUS:73feb1b665ad90296f7deab66005d89b
rScNaajFjkXFgapwyI1tHvCpDjiZS3IaLLNMLXhV85qe5GAufrNMNf2gpb685F8HgMjSzrExmUCir2CxkfaXk34A41y0wwxD8GlyYQWaBoiShKgrPgf09aq41OUobhS9O4xCKjDbyMUFn0X363T63a0VPKaQSSQxMe0motoM68hjY5cVYWRSXIEDjEk1m0pZ4fcizqUvo2FdXIYM:c3ba79e396c0b7aa8e2d13cb45919818
vFuidYLntVeGMuvHzxPZOAk2tQ6hoUTKqeTu3uHmgMST6eqwC6lgxw9Qc5YrPI16MKqG5XSczBVwFcS783dwpccs80KOzBKbc0H8OqaeRWAmZijYbnkrqmKodk2CLDEOucM8sNcaACmpKwdMJo4ZBEeEO77qAC5UEHSXlVXLna1Yw5B6jvWK00fFXdVoFQ9aoSYzDMbRN3FaZ768s:3bde04463a31e7085dd0e1774ec61c61
SJjJPOxSzMogLpvteVXQWEuQauF7qEq9ezJTe7CEJRKl65EaQsRDXbkXvQVMkcMpCm8GjBaTiL4fGzgnRXREZ2slIdYTfB8IN7eXyeGHPAMmZSAHGibwatHJxvsDWqbAnfnMkTj0kWd9FEGbMIxN25mpqETdVUD80QLaAV1KH4KcyhebPCOHx0XOvGSgrleh2QisB9284CbsJgupJ8:7799ac3e602d1c4c8a002a7e6f37abdb
X7ZKLkhtqyFvGxbOPCOIBIbD1LJl0ikOf9pRkwbAVGVr4X6KqLiSjklbVVnM4xraxGRH2T7OpT6jH3UYE2GNdSPZD2BxpSxNpfkiYiWe2SozM8ng1UT4CyT5rltH4RkjWurLDePvW4Vy29FTToOmNxh5IBCDiMM574HB8nWVhIkaHPUr4yDHWLDvc5YlIBhFvP6t2TfarpbZ4VgYkJw:37f72897a3ed2a4b69aebc3ec97b7454
7lZsoUhAtIIZ4nVogFtgXIhMD3SrdsNaEDtSoAiH9Qw44iiaOCgBbodEhWWlfzlJ3Frh5AP5hbZbk8bYAipBwt6EfSPki1UbwcjsCYnK0cLabNYBWd3iw0NCIsNRkxSGz2Z2QcCGfeHgSvIEzCNVsqn0J1IUoqrOjgG0j9woe3vXp4rOw5AOlOFVFdFTuXyD35tc4Qr9KNWZHeE394Il:dc663899cf0c94b3574ef60f1e5f99d0
Id6dgCXBpl4jgomb5DbFgXvN1ZH0Ufcsi8wpBjRRvL0C0dDVGErNsMrkCYbm3dFcc1sDb9lxLcZBftnMXEpghGqKurXovsrxkJ1mIcJjfyLkRZWFucLBI2c3k0iFiz2t8TFHWeR1CsbkhYPB0r3zj6ttWBZ51RpZBUxnZefsxhCF5i7Wz0mJXF3K7SFYAUO1FbfufkMDrf8nn5aNWmmJS:9404f789af0c16796f7a3446a805156c
fjPYPDoAhg5imqow3SO8Q1lL1Zxeb7WhhB5W5kmcrilDyA0riEqYwCAorYstVeA3GwPBH2EYbQszgiRpN8eaAEoRtGaeLB8iXNJvGejHUC70LPgoNkodPDV89VdLnbTbPtwlxGTi8Q9JvfY9pdc4GY3GK6hX71YMlkYJQHRPoQoKVdJbgM6MAZiaWqYUhMxtXl2DTJi7qxHcAgeGTajj0r:af088808d5053c870f7d13ee088e2609
kNH8xPKU8x601PAaMQxjSYqWu9XDjOVtrtrO82z6qvXhanhN4FXNtdAOdoim23GjmXYlQoh6J5dKjlneQbRafiOiwWvoQ1y3PnevBmrbhvLrGZvm1cm6l1eHOAWusko7yst0EL1M6D3CtpPkr2gDTlaxLWS4762vpllTWcwSGplZE09WSgpCrq03cJX9FQuV1GFOI1xfhI4Ly4yhaOJs4zl:c6afec8835087150168ade12743061f5
7896EPrqHXVmwMJBThwvGrUOryoKtNWrMVnhBFx9tilPUUhE2EZzVJdd8rNs5zJIkXQMsdLMmXs7HzBA3rZP13S0lvihUSP5FvIYzjAM6SKEic4ctURlNzbYVj5F1LBx79lxjM9gvKKDMFF6pnh2NiRysN4kovHluTJDwJJRjkkVPQR53ZXH7F5qiZAxl8IGSiafRKW0U7LAOtvIijfgPawX:511145575a549adec5d3c6d41cc2b24f
0XkcWTIF1IKIiwy3uat8uPGCZLIEM5rDTCFFleadNLMW8bPsb9qvPXYEzx8bsqFbSbH4fI8TjKFhLlzNkQ9ZDXu2ks4DIzFrBclHKjqTUV0v6q9h787AWrt6KoAjO5Af7MMHW2rGOi2Lz1Tw0RXM9gJJv9S94je1VRzHKQeoZ60oYJKOqxrpE0ZZZIYTR3LCKaalRfzHbqV0ZwFgTWwoMmdCU:011a7d2b95fd6de1159626c4a2f91038
2lBWX4xY5IFa8dcoY2elxl4kHjXTnpEfB52o0pdVYIVWMXbAQfMdrGoZQbJ3rd8S817ZhaK5JvSljUMZ0o3hVh6BjFuAJtjIlhxsh7nRs63BQGrGVkOGrK8Agt1PNanYhKgpHu70r0s8wJEhTiObTNCZ6tFKuSyCtFR10ORhFaf2KK9tTOvDrXsogYYBGn30SLRJqzQVzWN9wNtgsOJKC2YIQN:22f5d662e78df32d856672295b0ec4ad
9n13dKEVjUk0FKMjK9X4ffIPPaHEJohJB8CFI7rSRBInckwMkkHppf55gDzQRHqjFTPeQ6WxyuLAF8dPiK5yQ0tXtSEbpUkVDZZjVMHk6iLChOhppcN626UmPon48xQBnGL9iijfrUHyzOOPRrLJowV4K9YIwFaKbLJjUSpcCWB2BfHixjSMFEGfDvO00OA2p9baRrckdDdFJKO7TwJp0guuB8k:567dda05b1c486d04f5f3521a21a68ed
sNKk2Km2sD5DHzGd8gR60dFg4MAodzywc9H5JTY1nUuUj0Xi7EeYIJ4ClvQpVfMXejTO2IFfs00BQOJOjNCRnwkzS0PD6rraAaOtIkz0apsR42wdFYUTl5i4WYxSfOTGOyZnIoeJEwrypNCUCmEOiNI4B5MHJvdoU2CshG2M3kaiXC3qPy4XbDRDyua8QdnA6QTdmLPgVQpJjjZZHUNJnucMPcLv:444883ab5f180f20d065d2b35b421251
gZVcFEgSgWY1CeBMorBWcevqzx2fAEAGtlIYQPGWBvO3AgGOx7AzcVPCjIHKc7gwjPk0uQMVbrPCxmgUjGkms0OCzmcBjjXtY8jjOV5QdKiAWpl6WwjfmXHM9KejT2DI0MRFyMmBnv2aKd7xAgCN4ap4KteulH2bjKHInT9Bf1BP5zcvFF8AFougHyBS6tUf4sOhcNIhFaWBzZX4vWv1aP7HeyqaS:9d273fd62d29ce08bbc3a3cf485755ba
bgNtUUvytC4tpvTfiOA49r4OpCTOSTxT0adLUz9e13yhphmyVNtU5ozlQjZz3mjTnmu8LucDnAKDIX1uBkfxyEIfOyEHLNqY0LWB5Z5iqGMYtDiVOoidSRiwgN3SrkGhVtJQjFZzbBnv5wgJKpMt7VQezKWQVtYGMynV3cl5EzqzVx8wMLQKxxOX8ADTTsqwRDiLGtGaSWzEjy1WaIwx5bk3CONm73:07a912c4e96419d26a813bfd265b8515
IOxRp3bV3UHtoRRgERO0fK2pEu4iG1bfGzWVsXHmHfGWWn3q5Ihbj9qNulV1cX6ImSuFG1Re6n1TBV9xthyChPgB02s3PpC27XxEOfJLtBuVmuiGBG8Imfad7t6NIzFGm3a18KCsbXdohlUISSh4Xh8VA5y9u3FHWGzVg2diPhwXtgvbZD6NL4ycZXbkQGSDchpjatsPAPDTVYVLsSp3MO5CB6NhCwL:88020885a8ffa7927d7b474fc4cc4bc2
FD0PEk7kKMNuyCfaUYpNA4I3vEHD73fDx5i1ggbRTPbhhhi15yEwscqNHXhERm7eig6om855N7dVfmMaKhx2jNGRBOlj0jo9pknBjjxXgBImNlmnsKgCnNjPruYiEchTnvlwEIkLaSXN4a0mK7OYKXD2isaMvivIDH58ffJfYxTSxK58HJWixqaFJbh4kDCNaxMgDmmsJ5b7F66dGSl39lyInGCxa5ba:114db31384076641916279fa6d393dce
tNhWZtfzpgwVdsYJb9Dbl2JzycWz7YzqBHdr0jQgpc1tvP3xPwy0o8pNacc80Cps9sJ0Cp7SsYbNE4bjrAaFyqjzsmxjOMBO5aFypCgHqhlVcm5tM5Zbl8AEkYNZBeExe9VuB12S9dDbqyv3uk4FjvaukNjLSN9XMUheM9WLdqNtP8n9jiFsDFNOjWp1AoPd9WhLW4nZk0j0ZwZ9Evs815MaS22ihHlgu:b35330025e84362365b5751087eb3ed0
tSgn56yWfoMCeM6A45r0HsIzlQGS2NdlvjyrgXdmbQOgsLGnGYddhMCst9alMew8oVpuItGK9kRsVoFrdJvklYCEXDPAHcyVYNfx6M75XPoJ3T06tmgEaJ9YcPoJRMuGqA3mc1i0GwzAgpxzCE4Cd3qfIEPqhaWxBQJDHSueeTFl92KBxF3AzkG7PmoWmbkNRThpCBuHlZSkRtLeYFFnPcluoAhBBrfjbn:348323d4a99def0beed63b023e7de5a0
zDPTa0ISaqcMv0h0eW3qhiQpKhGPrI3QbIzcy8VzoXbJOiAT5u9dCPscXZSeyLVAjlCIjo7xb97QhyzdSyhkezn1p56OGSeQ3GodUmBWlyDtW2xErEPw42okXlYt32aXzo0kBr6NQA6Dsu7KYN6TFl3swRMpJMd913jtlg61G3vZnsAc5wVBIOU4vwk59n51gekslgkR9FHnY7pUual2P5Wbs67sk2jr7tJ:6e3a9dd3e19bd59c0e04c9ca2c97c844
iAU0ApxNev3YP4aHJ5uLorIGBAPSx9sGAcw0rUEvfxkUsks1gNCu5b1wBHfZHxw7zSXRDsdIPnt7yMZEpsZluQHVoNLVa7SAQGijYlRDzbBXNrs3Khe4YMQcpsXqpGR5cZf0bXuBYVpCdHwOPBJEepGuHuKX0rT2hYSiLdJAZY22wpHcQgGVwdf4yPRo7Ahep0n13XrSMkL9zjbgpH1LL6G9MnoJOwO4n2Vh:9253944163506b41cf1198c5d6fabff0
Qc0snBrMLtT01KCCH8CjmRT1dyVrrGy7JoqWQIzr1iiTTLlqJOAWvkNzJyR05fYFuerbNQsEZBnIczZMeqyAAcZaBH0wMP17Uj8x0r1PSfoVFeyJuW9VZZVqwMM9rE6bN5yDmqj5wyR1CfAW2AHRpt7M5KLneIpSDOmQ4vMqTtimyI8r9G8oZ60VGC8KansE2EkW0WdKglWFUVWTsV8iRZ3nr2YsghW8L7VCU:38b1828636d78a0656879fbdcdaceaef
Yd0aZwVLiF33DbLt5YKYM34TQ3vNOYyD1oERKpsTVmNpooIkdi8Qc3ASXVvBKU5B8AjJQCtBO6Q3lodoxc5z5viShDkRogjmGsWmUgOzdusoJwD6yywtUEc18nJmtSzZbmcWsQmvbFKL1eIpD4JnzLeyYOKiwa8ymalFRx0T2BuUF398XJlnlqL95mismgQzh24YpVIim32RW2PKB1XMizMdLUwY1cn85iXl3v:16e32cc55fdb615e2a742c55bd825e97
tfomWbfCLHuzkDYWGzH3ny5Bx0JkcCFvHUIuVOWxlgWvJKig9Q9nf5PCWoMYRikp2TJOxwbJDX4dydkyTkLYfrq2F3Rnb2C3LbI9xKI0xD3WGdkqnWftdvmJp4WqWokyqSyOs7FQAzC7DNoRztaCOnc3hYk4dutucR8vODcf2FdvircHKdaprdiy2tt5NMpQuob9ReeKJh5Rz8p9cqottxSlQbhtOxa8Lc8sgd2:fc0274f94cdbe05fb2070c7648f80c43
PkYxKWNJZ4Yinl0z12GlG4lFFnDXuGQaRFyrsb1h5Ppj0fIS8f4E9g9FuDtOa9ORFc9Xe14aGkjxqSfoYaSYgStAmMemMT4hW450VZ1saaPqSVEHVnv26Osjrxv3gqb3l6U6WLpWL5cuQH1CVnuR1N0jamcgDeaOaULWv0I7VLrCjj54wpMo3Dx3QAajEaYEUArgr0dcCvFLFAG1QjgJNDDt3DDyEr3pSkv9bzcD:33a9698201a079a6d3082097acc18c5e
VHfq8MiPVopyRSSLll9QD35ldrkfQmJCUOSTr1ydfNBXvkyHvynZrjbvAlBhyaJjPs26jq9o4BcpLBX70bXhkYDLa5Sy5rILKAITRI8L9a1LCOJsQwA0k3BlYkJUCivcI4VqCU1CUSenxNFda54lZwwNG6y8eUaNOW3gGUJrDNE0q94r4YCTkZxQVWPZGPDlBws8h1pKETBVTwcNlFxvuVcfHR5exypZVIX2AcD5W:a67da467d81629b486961a4321eeba6c
5QFrTjCpGYJr1PZJL36ajWVRJOiMhdydU4vEdYTKMt2E8RdKL9KUWwCvaK7iOwlyrG2vvM675YC4GFEhFf2rCvMC5KKK7W8pt1KOEGMzveu2KY9f42wwn9ZjKATImSXGKyke4XDq2XiCMiHHa44o4TxFjhewZ22zqdEl18BTVKlyST6TO08JJvfTCjgsb8SrcmC3k4N6EZvnIr6nh4WRq2ATbGcdFUvI7XChSfem4z:7b5c1dc4d45a30ee0376c6cefe51a5f8
ADRwQYrDGIv6rxDtA9eVHbJajsfnFaN5DurkjJOPhaMzxgTYfYKNZkN9CTn7T02nlkxuUc92cMsAScpY0ZBQafPDY2BITuV4Ejpplpho1zOKB3ysTZ8T5OnTHOsq8elDnBSzQ0NIpsiRlG94w8OrM2bjGtAExM8LdQbUHOtW7BEi8EdvCSMfKOpg8PLwsK7wr9hYNAKK2fT1JXmcFYhfDwmBcYXvyVRPUZDype9ijT9:1a6cd53a2d43387700c470d698975103
TGlWba3hEqt5thTPGExlpmPOCTNK3wKN2Wz4WtbqKum4C5J9A6kQj0uLJ7mDuWgnIFhvZiMzDZt6V35lZg296mKGkX9FJgSiMZEB8h1B6lyReu34AV3x8E3ir3N1awaMlEekM5MIgapluifV49i2DcbV6OM6aNImiMwUI8sO9IZuR5fL4oEyrFjnjVjkzSxHvUr3TKISiycZtiBow5MNAV1KHaV6TiDe3V8MvGuD4xto:64cfdf2f3ab4375ee87798ba50d0a0cf
FVC2QeGh0xRxyDuhMYmFJkh5RCyL0SZwDsokxUrni9LGD5OfT1KtbSoSlNubwky0CMaZxsdGsOcVKR1tICNKk2twGOy2YWSrzThXcktuZvgzd7TLzwmKpF7VjvN8iwpHfxvhHOCxaIWdGFP611GgwD2GZFE71UF7h0eOFHbFfYIMtoIlfesCIk8yPCwHm1EuSJ9ngbSmZrYiFwuLBMdaWbyCEUaRMFbveaIlbry1Jm9fJ:981458c11df1fdde3d81a304495ec71d
tQb6TbTVAlqlm8XRaiWbR7d9w4JJ4Oioukkel30MfhXrgK9gTViBTlBfgaOaOXo97zejs5VxcIPijOp3zxujJWoPW3QBQ5BOuFXNAIbDhQMqv2kUpF4ZriFdclESh6wCBkf2Thw0X9hi1r3hWX6d6Bh8NMRUIenaot37ApYoo6wfoqmaejeaKm9oZR9xVmyaFr86GWkVSG06Xd71nc2Xp2ceJbCuo0uThtPNf0yYwpVktT:b7f793f7ab9a7728b3e8eff67ccb1241
cGV5uaXxfwyHRMxcwEvciB2HppXbAgUCNFx8GkvMHTjpwGsTbNWjfPQv5DwvKgXoMkmjvHV3r4iOBBxCfkLkZsFUV1fw83KKeWUAuFtLACq13Ou8Y5iOoOzaGFwEy6fDTZ3dvmPVP6NIK7Hy3QdhuCh07EvvABYkqSoL54xK0ajA8QZ1x29RugIrK4NbvCmMkAngvLQlM0MKHBC4uBMOIl6jfKkAcwcnXqui2buE16fyyHt:d102e7e6faa10f88661b4400b23f45b0
S95xHgtQm41M7oPlc5FkYGFsxByWZ5gI5c5Dipkvjc7gqNrsJWDxt9PRAdEq9KY5MTyV9JgiLeo2RGlqtPNMOtuexYvxzktb4RX3r4b28qUQX57gKLTo4dts2PQrZ9tTRhM8bO1a4LQSHNYioIxjLRBEwiVlhf5ZmIXnmOohq4ZYIO6Wmu5YbxsxGeJNjFCwdqKq5YxmSnaq17dds91txkR3orGyW9U0qugliEy1RyhIwkMO:66a99a09b7fcf6be70cd37e4e982372a
kD8HOPA277qU7bKnvQzEkXv1mDAIOcnYGlvk1VdYTuIQvj3RZtlagH2sas1PvfD2qZdiLh7uBGaXP4EEnQFuoxNpQE4cKy41n89Yf6iGCjthdYMqFiKTFn8mS3osrjjEhjsmgBTJKdRN2t4xBFh7TgkB9yUqI4VPdd2jFLTgPAUHUO5mkcj3ituhSfyq9j6cnYmSA6ZfwJMgoIsyKCrTWM1ERpVQzRTMF6v62kboUOv8mNwXf:91fb9720ef1d47ab6e556eb378fda77d
OG1rxw8NhPcZzPvvaroltD57Y3kCq7H5Ezm1WlF31HSqmdMnv1yOuuMjnWLEUjApINhuYMoPk76wbIKwA8aUsN3GAFblOCAnfHI3jWJtUFQVeqiEpjpIWjfmOgxtIXGNvfHE2hYMNFyRVGmbPCJCLOOqVcJu0g8lmFQeNFRqaf8WLkXrNxTocyFXAfiqlgCyMjCpOjFOoDB0OphrM1GPpcCPhL6tRzRtIkJnte1IItznIGFlyb:5195b6e0e6c3767c500601a5dfb43225
0Nettl4qFLPm5owEBQJtohMNulkZwI1mm6FFhzWMbBZ6QmAh29BghdUBo5rLEjXQfsmnS8ZTAPPg1goTgP0nTLP8G7JbgwSmP5zx3phu4WBW2PF9vwme81cEZMG59oiOjHcdXjyS6ZEZFaY0wa4u27ZRJwNJLVnUtqYgAnYwcsmItayQbtb4r1M1nqAYbOJv4yBuLqROJ4m24KI64K0lBDdOTDDVhcgcBHxcYfQx92q3DY0yyRK:60a0c7d25131758b25cff7ec7e16df1b
0kNE3h8OpaVrBsPEhUkETdUNHJE68lGYvktprr4GsPYU8DofnzK7DuLb4f72qDRMOa15rVCKBquzUIE88e5BzHDuMAn244EI5wEmi6XJnh87QD6OI1g8yJsbaG34AyCvV6I33vDgDCej6aYE25DrFWsFsmATK3evZNpT8SZCk4Lg4ALWvfd11vxkHX3iQ8EGLUpKDfcN9OU3fwQABTstfPDNCxWjvrP6b4HEaJi9o33tpJURtcay:ebe7c20bb8a34c7df26ab805b9e7f317
SEcl2Yunf9kh4RL4rT3GNXZ2wKKPMLevfgG840laZvhUdTPvDJ1gwR9SrJx4kBpqIVoCLADBVLmZu1kXAc4WT3EaC24Muk325hvHHY9tAmjUduIEwCqfvVgXObA8lt0hBlP9aPtA2Cl6WUAimRoICuwhwWfIg6QIiwHjb0K4sW0fGqnT7CrAXNHjANRGKyfSUM2WDCQWoG1UXfDUHVluyTEYxmv7KAQudIhGbXsqukahpE2XqeiOY:2bb4f0b0fa89b62bed183e7d99a10a67
MDlYZjzpZUty1ZzPiP9tQhJlawBeEg2hKe6znvPcf9B7Z1M7GMrnT0zUn1YS8RZIVV88hObnNDkCuXA1zRejIE4VvTDtAs2me0lmFnAiQLLayc1YT68rB2dWLGgcZ9o40AhvNHEtigtHilw2iukJnnwZTC1JBgErGLNjihDrnXYv8logFyQTLcIvEA56Hzoda1nj9RBnEqJCBXJ7lpQWrpiWQdSnC7rDYFMYmebQuKjVxSSI7zvpOE:01c68c75b6751bd152a3557c7eed221a
bvHUIaS0DGvgvIkwoE6xpug83exptcTvxqg6qYWUvi0QQbc4F9sVuyUnChCWklhICnETbqDwzudfvfa1ftNzRxNkFf6PrenTsRDt86QxqUDMZED5XgvFE8P9eMOvQ1f8JIsIF96WTzyJ32FR90mDY1C2Dgok8tjI2CgxBcjlCIUwApdzgp34hvXLCbVBv59oxgLZjUaLtV8TabjgqcbyyZ911UsmPsAdyc2IX2jgNh0Oj9VAcXYBMXs:16f59e12fdcd8df5d8a90c5c57edea6e
DHLQn4hrtjj16cbdQi3ur3gR53GxFtP91rPemXwF6Gx3Iy7oH0JYTQGPJdcfwSenaUsmHOREkfxTDu7alHZ4nwKXzM2Vu78uRrHo5pSgugZXBXXMvNHI9hwzTokeljz3A6iwl0CF7ssyfgbBUI9TQwizKtEWD3Q4ZYqbPTGMcZbi6dJQLIAs4ISF2msvg8p6WG7CqDestgApj96VIwdD5V8X7rjdpy9c57ElBJDUP3aZ26KBsO4nz3bx:a50a2993a5530e7fbfef397125c3a021
kkBAJBCEy7PZGjJmcKc5GMwZqrdQkedupEVY5ottl8iSiS5lt7gZKDP0VSH5WKqceb0apjtBiCjqlfCEcIuMc9DXRaTELaGQbxQrHaTQD37P8zjb8ENaDhYliIQTInzKLg2igLYJFVpDlzEj3ikxp8JxHphfDGQeMInTkctPXIi87Ni0VsoaqXyYdF3GcKLPiYys0R7OqgNnTVeEdsFuf4jiAdPDNqiVfHe6pcLFipS2aWxee2zkXItoL:f8548f7fb500546af377ebd22d4f99a1
8RppZavwfrLh3gqiZijMFnRyN1lmkzXjgcy6dUtic4q5bGd1OmDkAlIecUqnjewqgVnkFGtHBKCDgGuVTY5k9DocoEpyIMOPy203yjB0T4t0AOMkDHuDbIFpdVNLysrWkhQJrhzbbT2CxEM1b6udF0sIM6kaoBXzJD80Ly2MhUfFoRwqORtjHMRjI2kXt8m3ClUdKMQSxVnMDK2hrlqYX88w1jjlhweJIYNjBt1ZefBIPtP6F6Vt5UPW3Z:92a125f42cd28f4fd7ba13e4fd871b1e
7bmcKU1n3tHViM0JkF2ZCyWnRR8xE1mC2zEDkwRePoZX0Pwalp0NNM1uuZSYR4qKuVdEhUIW9xKZc60nmQ1ZtSkNI2CpXSZhE3LMOjyNG8DJ5tXhaO7uxh75azkXia5N3GpHQNlnMOXI8KQ9pNtMVrHWQSKZSFCLcRjSvu689TH7un7J1qlMHSyoA9ejEGUGo3pKolIn5fkpsiYky9W6SlljlpsPWdmb7BLmNjzIpKYISMShMFdvRpE2FXH:4991538db21c5216e1012f76ddb89e66
r1UtZm4bZdAytam1TZJ0Bf4WWjrOPfm766qVikX8onxHoJy7z8Y0dSNZ25EHaRE7NUTW4GViTiQ8Se5hdT8gCL6uGB1HivEVfnIaUtjDBZCk3xr7hpeJ1a3yBU5KgAwlO4myov2Pku0eShbZxfypgsNIdJijaEvOzHdN26tcqjg8qiYOnn3tFH2IhaShEdW3K0GCWZFc9MbpUZ47dYrIwjhDk0LP3yIEof7a5Cd4eekZ4fX83eHpoOTzEvey:313da0449a11a1eb26b2b8e3a6fb7cf7
TM28SZ9OscJxhUmcZ9aTeH9tmTSReWfYz8WhXWWfYvC5gP86O8PSFPbSyTzDFFr4Decr0ZwPlZKrEJnjH31dI3V6NlzjQRekVhBM6XrhN2ziBNr9GjcedYbQaA0Ri414bsG8gYgTRF2iikiPulTxaUeal4RuZIo10VZ7J5QrAJAJUIoeT8CTTGT5BBpqaEi1p7YZ2FgteG3Yfi3YgvIZ1sVtTbKTFSKUQzKI4R28oWXt4QhbbQ138MmRNXBiG:d97118ed4ba29999ee6ac4654110fd87
c3mBEUvlNuzAhSuY0vZGnTEZ6mDTOLpRFBj9mFk0ZKq7sLVj7LPkuk9qxDabfpTKR2K3xVUnvakOL6YIIDS3OSkblkdqAW1hOBbcmVpIVAmxxa55EOZjx9lIjoZKbQRQiSSuD73pxgMVhHRLmGVJG6iQl7qmO72WQKhjIay6gaRnIyZU5LtBHBhTzx5dvYZby7k77J3eKLsi9h35SnwqPE9e25inT8pi5Af2997JKpSJxLFgZ1wEwwIorQCaPR:c49cdebc1186716be0a6839023499a15
jKhoDhnAQYQzydb3j0VGuh23vjKYkpFtZMItTWTAKz08D11MRMicUb6glG4vWzPMbo5VaOmKecJH3BkLetOYuK4FhY13nGFpUBbVgnwkp5Sjw2VAmzpGajch73blAHAl8B6ePsp4o7eL0ZccorJoALwyExJ44af3Cc8hvncJlglc5nekFOz6pVuUj4Pe4V7xX5EijGSuN3wJhBTMfiIv43fdYUhSGfgtbKCkhlFUe1tltdYzmxUgq0jeLrXhx3B:9bec1577495166361397f1f585f56f1d
yE3ILedwj5QvpTjO5GycXpm10DP80IzpctYOy2aHXQ3cAcQwJfywvKold3j4CJkF28jq0kYN1RQrUxNtDcQYM4aQYkLqTVVMT43JfRn6z4njr1DV4jJGeKm2UXiDJ4fsZ814Qf0g9epr6Td03Mw7nJ07x9A63GpSFhMmmDsmHIDDrGukjRiwq8uO7VK1s0KX7majQsvn1ZRiwbtGSB39znNx8ooreYebbflRyH5pxNyTORqxijwIxa6wpkNu9Swb:f9c885781f7ac8b7b079a8970c8e5b04
XHTlfPBCsZm6GCuPMRnaSjw7u9ruRNwFlfRqUiTc859EytkbbxbTHxQ2x8mfLIKWOscyBVBzRA4gUehWCiF9Fm12ke8VMIIq0LPrxgReHLkrPsehAJh6V8Z6cXSPwq6n2LFpRmuoXFGDXKLokSk6R92UWLzj1wwU719Odu30ZAuNa51lOchvbafYLF7D1tnZknEnHxexYzbZu2a84iu6iAUT5SnXbANmOhAlFFJt4KipDjoyrIuz8fj4YxS0XvcCD:73a96d30e4aced7f08c3486a57842ab5
DO88ns3YBirKwQ14ar3gBX4YFUM1XeuqSso6KhUcqcMMSEHTVAZnoUbTENKrS4IAnXwY5h0ljDY17wKTWA6ak7UPLv7tqFUDsgsnoizxLnpJJ0swqoxBmigYDdhTJ2n18FfmOEKqSZZrPiO6xcxKkEyNIGxhyK9xffKUJkar9ZJP8oLvq95AEtecZBkockLipl2Zv2hvSRbQ5NclM7MhrRjhjuvlE7uTimJEfQZXx0EsEGEgEhoWZx38RpklmE5vQE:f5137e7917ed7c27411e553d4f8678b8
ZVlOJIFe19LwggEu2tSWszcCLo7HJOcyznd9WIdOIPaPVF9NZSAhhCKTQHqqlsPLGSKsrO60t7FeDFSsnizvK9erGvISOntk64CoIze2wK60gPitXHOyHTPeenWSAg3xavLJVQBiry8xdgQ1Ovpmf4rjhe2IkW6kiIT3olbGJa3NhKEV6T8bOplWteuD0rO8Zy1eK3Kj3DWaerWaaVbPamBuqVXhdCf2176B1GLUayVePIeQtgvutWOJICQL4WDWTAn:75b9ec5825e241d928bf2573fd07f53b
K6Zv6nhb2QgJam4uFR8fpAW1vIuisVw3RloOywQrcW0c9UxFBvLRWxIi63rpONiwYxanTQEWDvYDgm8IIJpu7YD3RUivxrRmOSzxz4j3p8wvku3S4jd17HVPs4aQlS2ZBsn1nwUCvhYGBSow2snZZ8Eh3FYeoQ4Q9hHmEsP0zEw1nKnf3A5TzA1t6Pxlws1vAyH4RnUgRg75RkaLL5ubwlus1SDnauIATgvBtgHbnF64Q7Gs30tpcOI3wbgwwPXgViHf:f9c1fd490642c2994a27670f8a6ae583
