#!/usr/bin/perl -w
#
# Copyright 2012, Google Inc. All Rights Reserved.
#
# Licensed under the Apache License, Version 2.0 (the "License");
# you may not use this file except in compliance with the License.
# You may obtain a copy of the License at
#
#     http://www.apache.org/licenses/LICENSE-2.0
#
# Unless required by applicable law or agreed to in writing, software
# distributed under the License is distributed on an "AS IS" BASIS,
# WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
# See the License for the specific language governing permissions and
# limitations under the License.
#
# This example adds ad groups to a campaign. To get campaigns, run
# get_all_campaigns.pl.
#
# Tags: AdGroupService.mutate
# Author: David Torres <api.davidtorres@gmail.com>

use strict;
use lib "../../../lib";

use Google::Ads::AdWords::Client;
use Google::Ads::AdWords::Logging;
use Google::Ads::AdWords::v201402::AdGroup;
use Google::Ads::AdWords::v201402::AdGroupOperation;
use Google::Ads::AdWords::v201402::BiddingStrategyConfiguration;
use Google::Ads::AdWords::v201402::CpcBid;
use Google::Ads::AdWords::v201402::Money;
use Google::Ads::AdWords::v201402::TargetingSetting;
use Google::Ads::AdWords::v201402::TargetingSettingDetail;

use Cwd qw(abs_path);
use Data::Uniqid qw(uniqid);

# Replace with valid values of your account.
my $campaign_id = "INSERT_CAMPAIGN_ID_HERE";

# Example main subroutine.
sub add_ad_groups {
  my $client = shift;
  my $campaign_id = shift;

  my $num_ad_groups = 2;
  my @operations = ();
  for (my $i = 0; $i < $num_ad_groups; $i++) {
    # Create ad group.
    my $ad_group = Google::Ads::AdWords::v201402::AdGroup->new({
      name => "Earth to Mars Cruises #" . uniqid(),
      campaignId => $campaign_id,
      biddingStrategyConfiguration =>
          Google::Ads::AdWords::v201402::BiddingStrategyConfiguration->new({
            bids => [
              Google::Ads::AdWords::v201402::CpcBid->new({
                bid => Google::Ads::AdWords::v201402::Money->new({
                  microAmount => 1000000
                }),
              }),
            ]
          }),
      # Additional properties (non-required).
      status => "ENABLED",
      settings => [
        # Targetting restriction settings - these setting only affect serving
        # for the Display Network.
        Google::Ads::AdWords::v201402::TargetingSetting->new({
          details => [
            # Restricting to serve ads that match your ad group placements.
            Google::Ads::AdWords::v201402::TargetingSettingDetail->new({
              criterionTypeGroup => "PLACEMENT",
              targetAll => 1
            }),
            # Using your ad group verticals only for bidding.
            Google::Ads::AdWords::v201402::TargetingSettingDetail->new({
              criterionTypeGroup => "VERTICAL",
              targetAll => 0
            })
          ]
        })
      ]
    });
    # Create operation.
    my $ad_group_operation =
        Google::Ads::AdWords::v201402::AdGroupOperation->new({
          operator => "ADD",
          operand => $ad_group
        });
    push @operations, $ad_group_operation;
  }

  # Add ad groups.
  my $result = $client->AdGroupService()->mutate({
    operations => \@operations
  });

  # Display ad groups.
  foreach my $ad_group (@{$result->get_value()}) {
    printf "Ad group with name \"%s\" and id \"%d\" was added.\n",
           $ad_group->get_name(), $ad_group->get_id();
  }

  return 1;
}

# Don't run the example if the file is being included.
if (abs_path($0) ne abs_path(__FILE__)) {
  return 1;
}

# Log SOAP XML request, response and API errors.
Google::Ads::AdWords::Logging::enable_all_logging();

# Get AdWords Client, credentials will be read from ~/adwords.properties.
my $client = Google::Ads::AdWords::Client->new({version => "v201402"});

# By default examples are set to die on any server returned fault.
$client->set_die_on_faults(1);

# Call the example
add_ad_groups($client, $campaign_id);
