#!/usr/bin/env perl

use strict;
use warnings;

use Getopt::Long;
use Pod::Usage;
use IO::All -utf8;

use HON::I18N::Converter;

=head1 NAME

build-properties-INI-file.pl

=head1 VERSION

Version 0.02

=head1 DESCRIPTION

Convert an excel i18n file to ini format

=head1 USAGE

./build-properties-INI-file.pl --help

./build-properties-INI-file.pl --i18n=path/to/my/file.xls --output=/tmp/ini

=head1 REQUIRED ARGUMENTS

=over 2

=item --i18n=path/to/my/file.xls

The excel file with all the language label -> traduction

=item --output=/tmp/ini

The destination folder

=back

=cut

our $VERSION = '0.02';

my ( $help, $i18n, $output );

GetOptions(
  'i18n=s'   => \$i18n,
  'output=s' => \$output,
  'help'     => \$help,
) || pod2usage(2);

if ( $help || !$i18n || !$output ) {
  pod2usage(1);
  exit 0;
}

my $parser = HON::I18N::Converter->new( excel => $i18n );

my $header = <<"EOS";
; Automatically generated by the HON-I18N-Converter project
; Please do not modify this file directly!!

EOS

$parser->build_properties_file( 'INI', $output, $header );

=head1 AUTHOR

Samia Chahlal, C<< <samia.chahlal at yahoo.com> >>

=head1 BUGS AND LIMITATIONS

Please report any bugs or feature requests to C<bug-hon-i18n-converter at rt.cpan.org>, or through
the web interface at L<http://rt.cpan.org/NoAuth/ReportBug.html?Queue=HON-I18N-Converter>.  I will be notified, and then you'll
automatically be notified of progress on your bug as I make changes.

=head1 SUPPORT

You can find documentation for this module with the perldoc command.

    perldoc HON::I18N::Converter


You can also look for information at:

=over 4

=item * RT: CPAN's request tracker (report bugs here)

L<http://rt.cpan.org/NoAuth/Bugs.html?Dist=HON-I18N-Converter>

=item * AnnoCPAN: Annotated CPAN documentation

L<http://annocpan.org/dist/HON-I18N-Converter>

=item * CPAN Ratings

L<http://cpanratings.perl.org/d/HON-I18N-Converter>

=item * Search CPAN

L<http://search.cpan.org/dist/HON-I18N-Converter/>

=back


=head1 ACKNOWLEDGEMENTS


=head1 LICENSE AND COPYRIGHT

Copyright 2013 Samia Chahlal.

This program is free software; you can redistribute it and/or modify it
under the terms of either: the GNU General Public License as published
by the Free Software Foundation; or the Artistic License.

See http://dev.perl.org/licenses/ for more information.

=cut
