use Config;
use File::Basename < qw(&basename &dirname);
use File::Spec;
use Cwd;

my $origdir = cwd;
chdir dirname($^PROGRAM_NAME);
my $file = basename($^PROGRAM_NAME, '.PL');
$file =~ s!_(pm)$!.$1!i;

my $useConfig;
my $Config_archname;
my $Config_version;
my $Config_inc_version_list;

# Expand the variables only if explicitly requested because
# otherwise relocating Perl becomes much harder.

if (env::var('PERL_BUILD_EXPAND_CONFIG_VARS')) {
    $useConfig = '';
    $Config_archname = qq('$(config_value("archname"))');
    $Config_version  = qq('$(config_value("version"))');
    my @Config_inc_version_list =
	reverse split m/ /, config_value("inc_version_list");
    $Config_inc_version_list =
	(nelems @Config_inc_version_list) ??
	    qq($(join ' ',@Config_inc_version_list)) !! q(());
} else {
    $useConfig = 'use Config;';
    $Config_archname = q($(config_value('archname')));
    $Config_version  = q($(config_value('version')));
    $Config_inc_version_list =
	      q(reverse split m/ /, config_value('inc_version_list'));
}
 
open my $out,">", "$file" or die "Can't create $file: $^OS_ERROR";
 
print $^STDOUT, "Extracting $file (with variable substitutions)\n";
 
# In this section, perl variables will be expanded during extraction.
# You can use %Config{...} to use Configure variables.
 
print $out, <<"!GROK!THIS!";
package lib;

# THIS FILE IS AUTOMATICALLY GENERATED FROM lib_pm.PL.
# ANY CHANGES TO THIS FILE WILL BE OVERWRITTEN BY THE NEXT PERL BUILD.

$useConfig

my \$archname         = $Config_archname;
my \$version          = $Config_version;
my \@inc_version_list = $Config_inc_version_list;

!GROK!THIS!
print $out, <<'!NO!SUBS!';

our @ORIG_INCLUDE_PATH = $^INCLUDE_PATH;	# take a handy copy of 'original' value
our $VERSION = '0.5565';
my $Is_MacOS = $^OS_NAME eq 'MacOS';
my $Mac_FS;
if ($Is_MacOS) {
	require File::Spec;
	$Mac_FS = try { require Mac::FileSpec::Unixish };
}

sub import {
    shift;

    my %names;
    foreach (reverse @_) {
	my $path = $_;		# we'll be modifying it, so break the alias
	if ($path eq '') {
	    require Carp;
	    Carp::carp("Empty compile time value given to use lib");
	}

	$path = _nativize($path);

	if (-e $path && ! -d _) {
	    require Carp;
	    Carp::carp("Parameter to use lib must be directory, not file");
	}
	unshift($^INCLUDE_PATH, $path);
	# Add any previous version directories we found at configure time
	foreach my $incver (@inc_version_list)
	{
	    my $dir = $Is_MacOS
		?? File::Spec->catdir( $path, $incver )
		!! "$path/$incver";
	    unshift($^INCLUDE_PATH, $dir) if -d $dir;
	}
	# Put a corresponding archlib directory in front of $path if it
	# looks like $path has an archlib directory below it.
	my @($arch_auto_dir, $arch_dir, $version_dir, $version_arch_dir)
	    = _get_dirs($path);
	unshift($^INCLUDE_PATH, $arch_dir)         if -d $arch_auto_dir;
	unshift($^INCLUDE_PATH, $version_dir)      if -d $version_dir;
	unshift($^INCLUDE_PATH, $version_arch_dir) if -d $version_arch_dir;
    }

    # remove trailing duplicates
    $^INCLUDE_PATH = grep { ++%names{+$_} == 1 }, $^INCLUDE_PATH;
    return;
}


sub unimport {
    shift;

    my %names;
    foreach (@_) {
	my $path = _nativize($_);

	my @($arch_auto_dir, $arch_dir, $version_dir, $version_arch_dir)
	    = _get_dirs($path);
	++%names{+$path};
	++%names{+$arch_dir}         if -d $arch_auto_dir;
	++%names{+$version_dir}      if -d $version_dir;
	++%names{+$version_arch_dir} if -d $version_arch_dir;
    }

    # Remove ALL instances of each named directory.
    $^INCLUDE_PATH = grep { !exists %names{$_} }, $^INCLUDE_PATH;
    return;
}

sub _get_dirs {
    my @($dir) = @_;
    my($arch_auto_dir, $arch_dir, $version_dir, $version_arch_dir);

    # we could use this for all platforms in the future, but leave it
    # Mac-only for now, until there is more time for testing it.
    if ($Is_MacOS) {
	$arch_auto_dir    = File::Spec->catdir( $dir, $archname, 'auto' );
	$arch_dir         = File::Spec->catdir( $dir, $archname, );
	$version_dir      = File::Spec->catdir( $dir, $version );
	$version_arch_dir = File::Spec->catdir( $dir, $version, $archname );
    } else {
	$arch_auto_dir    = "$dir/$archname/auto";
	$arch_dir         = "$dir/$archname";
	$version_dir      = "$dir/$version";
	$version_arch_dir = "$dir/$version/$archname";
    }
    return @($arch_auto_dir, $arch_dir, $version_dir, $version_arch_dir);
}

sub _nativize {
    my @($dir) = @_;

    if ($Is_MacOS && $Mac_FS && ! -d $dir) {
	$dir = Mac::FileSpec::Unixish::nativize($dir);
	$dir .= ":" unless $dir =~ m/:$/;
    }

    return $dir;
}

1;
__END__

=head1 NAME

lib - manipulate $^INCLUDE_PATH at compile time

=head1 SYNOPSIS

    use lib LIST;

    no lib LIST;

=head1 DESCRIPTION

This is a small simple module which simplifies the manipulation of $^INCLUDE_PATH
at compile time.

It is typically used to add extra directories to perl's search path so
that later C<use> or C<require> statements will find modules which are
not located on perl's default search path.

=head2 Adding directories to $^INCLUDE_PATH

The parameters to C<use lib> are added to the start of the perl search
path. Saying

    use lib LIST;

is I<almost> the same as saying

    BEGIN { unshift($^INCLUDE_PATH, LIST) }

For each directory in LIST (called $dir here) the lib module also
checks to see if a directory called $dir/$archname/auto exists.
If so the $dir/$archname directory is assumed to be a corresponding
architecture specific directory and is added to $^INCLUDE_PATH in front of $dir.

The current value of C<$archname> can be found with this command:

    perl -V:archname

To avoid memory leaks, all trailing duplicate entries in $^INCLUDE_PATH are
removed.

=head2 Deleting directories from $^INCLUDE_PATH

You should normally only add directories to $^INCLUDE_PATH.  If you need to
delete directories from $^INCLUDE_PATH take care to only delete those which you
added yourself or which you are certain are not needed by other modules
in your script.  Other modules may have added directories which they
need for correct operation.

The C<no lib> statement deletes all instances of each named directory
from $^INCLUDE_PATH.

For each directory in LIST (called $dir here) the lib module also
checks to see if a directory called $dir/$archname/auto exists.
If so the $dir/$archname directory is assumed to be a corresponding
architecture specific directory and is also deleted from $^INCLUDE_PATH.

=head2 Restoring original $^INCLUDE_PATH

When the lib module is first loaded it records the current value of $^INCLUDE_PATH
in an array C<@lib::ORIG_INCLUDE_PATH. To restore $^INCLUDE_PATH to that value you
can say

    $^INCLUDE_PATH = @lib::ORIG_INC;

=head1 CAVEATS

In order to keep lib.pm small and simple, it only works with Unix
filepaths.  This doesn't mean it only works on Unix, but non-Unix
users must first translate their file paths to Unix conventions.

    # VMS users wanting to put [.stuff.moo] into 
    # their $^INCLUDE_PATH would write
    use lib 'stuff/moo';

=head1 NOTES

In the future, this module will likely use File::Spec for determining
paths, as it does now for Mac OS (where Unix-style or Mac-style paths
work, and Unix-style paths are converted properly to Mac-style paths
before being added to $^INCLUDE_PATH).

=head1 SEE ALSO

FindBin - optional module which deals with paths relative to the source file.

=head1 AUTHOR

Tim Bunce, 2nd June 1995.

=cut
!NO!SUBS!

close $out or die "Can't close $file: $^OS_ERROR";
chdir $origdir;
