#!/usr/bin/perl
use Test::More tests => 18061;
BEGIN { use_ok('Math::Telephony::ErlangB') };
use Math::Telephony::ErlangB qw(:all);

my @gos = split /\t/, <DATA>;
shift @gos;      # Eliminate "n"
chomp($gos[-1]); # Eliminate "\n"
while (<DATA>) {
   chomp;
   my @traffic = split /\t/;
   my $servers = shift @traffic; # First element is # of servers
   my $factor = ($servers < 200) ? 0.995 : 0.999;
   foreach my $i (0 .. $#gos) {
      cmp_ok(abs(gos($traffic[$i], $servers) - $gos[$i]),
         "<", 1e-4, "gos($traffic[$i], $servers) -> $gos[$i]");
      cmp_ok(abs(traffic($servers, $gos[$i], 1e-10) - $traffic[$i]),
             "<", 0.03, "traffic($servers, $gos[$i]) -> $traffic[$i]");

      # Scale traffic a bit down, just to take into accunt roundoff
      # when evaluating necessary number of servers
      $traffic[$i] *= $factor;
      is(servers($traffic[$i], $gos[$i]), $servers,
         "servers($traffic[$i], $gos[$i]) -> $servers");
   }
}

__DATA__
n	0.00001	0.00005	0.0001	0.0005	0.001	0.002	0.003	0.004	0.005	0.006	0.007	0.008	0.009	0.01	0.02	0.03	0.05	0.1	0.2	0.4
1	0.00001	0.00005	0.0001	0.0005	0.001	0.002	0.00301	0.00402	0.00503	0.00604	0.00705	0.00806	0.00908	0.0101	0.02041	0.03093	0.05263	0.11111	0.25	0.66667	1
2	0.00448	0.01005	0.01425	0.03213	0.04576	0.06534	0.08064	0.09373	0.1054	0.11608	0.126	0.13532	0.14416	0.15259	0.22347	0.28155	0.38132	0.59543	1	2	2
3	0.0398	0.06849	0.08683	0.1517	0.19384	0.24872	0.28851	0.32099	0.349	0.37395	0.39664	0.41757	0.43711	0.45549	0.60221	0.71513	0.8994	1.2708	1.9299	3.4798	3
4	0.12855	0.19554	0.23471	0.36236	0.43927	0.53503	0.60209	0.65568	0.7012	0.74124	0.77729	0.81029	0.84085	0.86942	1.0923	1.2589	1.5246	2.0454	2.9452	5.021	4
5	0.27584	0.38851	0.45195	0.64857	0.76212	0.89986	0.99446	1.0692	1.132	1.187	1.2362	1.281	1.3223	1.3608	1.6571	1.8752	2.2185	2.8811	4.0104	6.5955	5
6	0.47596	0.63923	0.72826	0.99567	1.1459	1.3252	1.4468	1.5421	1.6218	1.6912	1.7531	1.8093	1.861	1.909	2.2759	2.5431	2.9603	3.7584	5.1086	8.1907	6
7	0.72378	0.93919	1.0541	1.3922	1.5786	1.7984	1.9463	2.0614	2.1575	2.2408	2.3149	2.382	2.4437	2.5009	2.9354	3.2497	3.7378	4.6662	6.2302	9.7998	7
8	1.0133	1.2816	1.4219	1.8298	2.0513	2.3106	2.4837	2.6181	2.7299	2.8266	2.9125	2.9902	3.0615	3.1276	3.6271	3.9865	4.543	5.5971	7.3692	11.419	8
9	1.3391	1.6595	1.8256	2.3016	2.5575	2.8549	3.0526	3.2057	3.3326	3.4422	3.5395	3.6274	3.708	3.7825	4.3447	4.7479	5.3702	6.5464	8.5217	13.045	9
10	1.697	2.0689	2.2601	2.8028	3.092	3.4265	3.648	3.819	3.9607	4.0829	4.1911	4.2889	4.3784	4.4612	5.084	5.5294	6.2157	7.5106	9.685	14.677	10
11	2.0849	2.5059	2.7216	3.3294	3.6511	4.0215	4.2661	4.4545	4.6104	4.7447	4.8637	4.9709	5.0691	5.1599	5.8415	6.328	7.0764	8.4871	10.857	16.314	11
12	2.4958	2.9671	3.2072	3.8781	4.2314	4.6368	4.9038	5.1092	5.2789	5.425	5.5543	5.6708	5.7774	5.876	6.6147	7.141	7.9501	9.474	12.036	17.954	12
13	2.9294	3.45	3.7136	4.4465	4.8306	5.27	5.5588	5.7807	5.9638	6.1214	6.2607	6.3863	6.5011	6.6072	7.4015	7.9667	8.8349	10.47	13.222	19.598	13
14	3.3834	3.9523	4.2388	5.0324	5.4464	5.919	6.2291	6.467	6.6632	6.832	6.9811	7.1155	7.2382	7.3517	8.2003	8.8035	9.7295	11.473	14.413	21.243	14
15	3.8559	4.4721	4.7812	5.6339	6.0772	6.5822	6.913	7.1665	7.3755	7.5552	7.7139	7.8568	7.9874	8.108	9.0096	9.65	10.633	12.484	15.608	22.891	15
16	4.3453	5.0079	5.339	6.2496	6.7215	7.2582	7.6091	7.878	8.0995	8.2898	8.4579	8.6092	8.7474	8.875	9.8284	10.505	11.544	13.5	16.807	24.541	16
17	4.8502	5.5583	5.911	6.8782	7.3781	7.9457	8.3164	8.6003	8.834	9.0347	9.2119	9.3714	9.5171	9.6516	10.656	11.368	12.461	14.522	18.01	26.192	17
18	5.3693	6.122	6.4959	7.5186	8.0459	8.6437	9.0339	9.3324	9.578	9.7889	9.9751	10.143	10.296	10.437	11.491	12.238	13.385	15.548	19.216	27.844	18
19	5.9016	6.698	7.0927	8.1698	8.7239	9.3515	9.7606	10.073	10.331	10.552	10.747	10.922	11.082	11.23	12.333	13.115	14.315	16.579	20.424	29.498	19
20	6.446	7.2854	7.7005	8.831	9.4115	10.068	10.496	10.823	11.092	11.322	11.526	11.709	11.876	12.031	13.182	13.997	15.249	17.613	21.635	31.152	20
21	7.0017	7.8834	8.3186	9.5014	10.108	10.793	11.239	11.58	11.86	12.1	12.312	12.503	12.677	12.838	14.036	14.885	16.189	18.651	22.848	32.808	21
22	7.568	8.4926	8.9462	10.18	10.812	11.525	11.989	12.344	12.635	12.885	13.105	13.303	13.484	13.651	14.896	15.778	17.132	19.692	24.064	34.464	22
23	8.1443	9.1095	9.5826	10.868	11.524	12.265	12.746	13.114	13.416	13.676	13.904	14.11	14.297	14.47	15.761	16.675	18.08	20.737	25.281	36.121	23
24	8.7298	9.7351	10.227	11.562	12.243	13.011	13.51	13.891	14.204	14.472	14.709	14.922	15.116	15.295	16.631	17.577	19.031	21.784	26.499	37.779	24
25	9.324	10.369	10.88	12.264	12.969	13.763	14.279	14.673	14.997	15.274	15.519	15.739	15.939	16.125	17.505	18.483	19.985	22.833	27.72	39.437	25
26	9.9265	11.01	11.54	12.972	13.701	14.522	15.054	15.461	15.795	16.081	16.334	16.561	16.768	16.959	18.383	19.392	20.943	23.885	28.941	41.096	26
27	10.537	11.659	12.207	13.686	14.439	15.285	15.835	16.254	16.598	16.893	17.153	17.387	17.601	17.797	19.265	20.305	21.904	24.939	30.164	42.755	27
28	11.154	12.314	12.88	14.406	15.182	16.054	16.62	17.051	17.406	17.709	17.977	18.218	18.438	18.64	20.15	21.221	22.867	25.995	31.388	44.414	28
29	11.779	12.976	13.56	15.132	15.93	16.828	17.41	17.853	18.218	18.53	18.805	19.053	19.279	19.487	21.039	22.14	23.833	27.053	32.614	46.074	29
30	12.417	13.644	14.246	15.863	16.684	17.606	18.204	18.66	19.034	19.355	19.637	19.891	20.123	20.337	21.932	23.062	24.802	28.113	33.84	47.735	30
31	13.054	14.318	14.937	16.599	17.442	18.389	19.002	19.47	19.854	20.183	20.473	20.734	20.972	21.191	22.827	23.987	25.773	29.174	35.067	49.395	31
32	13.697	14.998	15.633	17.34	18.205	19.176	19.805	20.284	20.678	21.015	21.312	21.58	21.823	22.048	23.725	24.914	26.746	30.237	36.295	51.056	32
33	14.346	15.682	16.335	18.085	18.972	19.966	20.611	21.102	21.505	21.85	22.155	22.429	22.678	22.909	24.626	25.844	27.721	31.301	37.524	52.718	33
34	15.001	16.372	17.041	18.835	19.743	20.761	21.421	21.923	22.336	22.689	23.001	23.281	23.536	23.772	25.529	26.776	28.698	32.367	38.754	54.379	34
35	15.66	17.067	17.752	19.589	20.517	21.559	22.234	22.748	23.169	23.531	23.849	24.136	24.397	24.638	26.435	27.711	29.677	33.434	39.985	56.041	35
36	16.325	17.766	18.468	20.347	21.296	22.361	23.05	23.575	24.006	24.376	24.701	24.994	25.261	25.507	27.343	28.647	30.657	34.503	41.216	57.703	36
37	16.995	18.47	19.188	21.108	22.078	23.166	23.87	24.406	24.846	25.223	25.556	25.854	26.127	26.378	28.254	29.585	31.64	35.572	42.448	59.365	37
38	17.669	19.178	19.911	21.873	22.864	23.974	24.692	25.24	25.689	26.074	26.413	26.718	26.996	27.252	29.166	30.526	32.624	36.643	43.68	61.028	38
39	18.348	19.89	20.64	22.642	23.652	24.785	25.518	26.076	26.534	26.926	27.272	27.583	27.867	28.129	30.081	31.468	33.609	37.715	44.913	62.69	39
40	19.031	20.606	21.372	23.414	24.444	25.599	26.346	26.915	27.382	27.782	28.134	28.451	28.741	29.007	30.997	32.412	34.596	38.787	46.147	64.353	40
41	19.718	21.326	22.107	24.189	25.239	26.416	27.177	27.756	28.232	28.64	28.999	29.322	29.616	29.888	31.916	33.357	35.584	39.861	47.381	66.016	41
42	20.409	22.049	22.846	24.967	26.037	27.235	28.01	28.6	29.085	29.5	29.866	30.194	30.494	30.771	32.836	34.305	36.574	40.936	48.616	67.679	42
43	21.104	22.776	23.587	25.748	26.837	28.057	28.846	29.447	29.94	30.362	30.734	31.069	31.374	31.656	33.758	35.253	37.565	42.011	49.851	69.342	43
44	21.803	23.507	24.333	26.532	27.641	28.882	29.684	30.295	30.797	31.227	31.605	31.946	32.256	32.543	34.682	36.203	38.557	43.088	51.086	71.006	44
45	22.505	24.24	25.081	27.319	28.447	29.708	30.525	31.146	31.656	32.093	32.478	32.824	33.14	33.432	35.607	37.155	39.55	44.165	52.322	72.669	45
46	23.211	24.977	25.833	28.109	29.255	30.538	31.367	31.999	32.517	32.962	33.353	33.705	34.026	34.322	36.534	38.108	40.545	45.243	53.559	74.333	46
47	23.921	25.717	26.587	28.901	30.066	31.369	32.212	32.854	33.381	33.832	34.23	34.587	34.913	35.215	37.462	39.062	41.54	46.322	54.796	75.997	47
48	24.633	26.46	27.344	29.696	30.879	32.203	33.059	33.711	34.246	34.704	35.108	35.471	35.803	36.109	38.392	40.018	42.537	47.401	56.033	77.66	48
49	25.349	27.206	28.104	30.493	31.694	33.039	33.908	34.57	35.113	35.578	35.988	36.357	36.694	37.004	39.323	40.975	43.534	48.481	57.27	79.324	49
50	26.067	27.954	28.867	31.292	32.512	33.876	34.759	35.431	35.982	36.454	36.87	37.245	37.586	37.901	40.255	41.933	44.533	49.562	58.508	80.988	50
51	26.789	28.706	29.632	32.094	33.332	34.716	35.611	36.293	36.852	37.331	37.754	38.134	38.48	38.8	41.189	42.892	45.533	50.644	59.746	82.652	51
52	27.513	29.459	30.4	32.898	34.153	35.558	36.466	37.157	37.724	38.211	38.639	39.024	39.376	39.7	42.124	43.852	46.533	51.726	60.985	84.317	52
53	28.241	30.216	31.17	33.704	34.977	36.401	37.322	38.023	38.598	39.091	39.526	39.916	40.273	40.602	43.06	44.813	47.534	52.808	62.224	85.981	53
54	28.971	30.975	31.942	34.512	35.803	37.247	38.18	38.891	39.474	39.973	40.414	40.81	41.171	41.505	43.997	45.776	48.536	53.891	63.463	87.645	54
55	29.703	31.736	32.717	35.322	36.631	38.094	39.04	39.76	40.351	40.857	41.303	41.705	42.071	42.409	44.936	46.739	49.539	54.975	64.702	89.31	55
56	30.438	32.5	33.494	36.134	37.46	38.942	39.901	40.63	41.229	41.742	42.194	42.601	42.972	43.315	45.875	47.703	50.543	56.059	65.942	90.974	56
57	31.176	33.266	34.273	36.948	38.291	39.793	40.763	41.502	42.109	42.629	43.087	43.499	43.875	44.222	46.816	48.669	51.548	57.144	67.181	92.639	57
58	31.916	34.034	35.055	37.764	39.124	40.645	41.628	42.376	42.99	43.516	43.98	44.398	44.778	45.13	47.758	49.635	52.553	58.229	68.421	94.303	58
59	32.659	34.804	35.838	38.581	39.959	41.498	42.493	43.251	43.873	44.406	44.875	45.298	45.683	46.039	48.7	50.602	53.559	59.315	69.662	95.968	59
60	33.404	35.577	36.623	39.401	40.795	42.353	43.36	44.127	44.757	45.296	45.771	46.199	46.589	46.95	49.644	51.57	54.566	60.401	70.902	97.633	60
61	34.151	36.351	37.411	40.222	41.633	43.21	44.229	45.005	45.642	46.188	46.669	47.102	47.497	47.861	50.589	52.539	55.573	61.488	72.143	99.297	61
62	34.9	37.127	38.2	41.045	42.472	44.068	45.099	45.884	46.528	47.081	47.567	48.005	48.405	48.774	51.534	53.508	56.581	62.575	73.384	100.96	62
63	35.651	37.906	38.991	41.869	43.313	44.927	45.97	46.764	47.416	47.975	48.467	48.91	49.314	49.688	52.481	54.478	57.59	63.663	74.625	102.63	63
64	36.405	38.686	39.784	42.695	44.156	45.788	46.843	47.646	48.305	48.87	49.368	49.816	50.225	50.603	53.428	55.45	58.599	64.75	75.866	104.29	64
65	37.16	39.468	40.579	43.523	45	46.65	47.716	48.528	49.195	49.766	50.27	50.723	51.137	51.518	54.376	56.421	59.609	65.839	77.108	105.96	65
66	37.918	40.252	41.375	44.352	45.845	47.513	48.591	49.412	50.086	50.664	51.173	51.631	52.049	52.435	55.325	57.394	60.619	66.927	78.35	107.62	66
67	38.677	41.038	42.173	45.183	46.692	48.378	49.467	50.297	50.978	51.562	52.077	52.54	52.963	53.353	56.275	58.367	61.63	68.016	79.592	109.29	67
68	39.439	41.825	42.973	46.015	47.54	49.243	50.345	51.183	51.872	52.462	52.982	53.45	53.877	54.272	57.226	59.341	62.642	69.106	80.834	110.95	68
69	40.202	42.615	43.775	46.848	48.389	50.11	51.223	52.071	52.766	53.362	53.888	54.361	54.793	55.191	58.177	60.316	63.654	70.196	82.076	112.62	69
70	40.967	43.405	44.578	47.683	49.239	50.979	52.103	52.959	53.662	54.264	54.795	55.273	55.709	56.112	59.129	61.291	64.667	71.286	83.318	114.28	70
71	41.734	44.198	45.382	48.519	50.091	51.848	52.984	53.848	54.558	55.166	55.703	56.186	56.626	57.033	60.082	62.267	65.68	72.376	84.561	115.95	71
72	42.502	44.992	46.188	49.357	50.944	52.718	53.865	54.739	55.455	56.07	56.612	57.099	57.545	57.956	61.036	63.244	66.694	73.467	85.803	117.61	72
73	43.273	45.787	46.996	50.195	51.799	53.59	54.748	55.63	56.354	56.974	57.522	58.014	58.464	58.879	61.99	64.221	67.708	74.558	87.046	119.28	73
74	44.045	46.585	47.805	51.035	52.654	54.463	55.632	56.522	57.253	57.88	58.432	58.93	59.384	59.803	62.945	65.199	68.723	75.649	88.289	120.94	74
75	44.818	47.383	48.615	51.877	53.511	55.337	56.517	57.415	58.153	58.786	59.344	59.846	60.304	60.728	63.9	66.177	69.738	76.741	89.532	122.61	75
76	45.593	48.183	49.427	52.719	54.369	56.211	57.402	58.31	59.054	59.693	60.256	60.763	61.226	61.653	64.857	67.156	70.753	77.833	90.776	124.27	76
77	46.37	48.985	50.24	53.563	55.227	57.087	58.289	59.205	59.956	60.601	61.169	61.681	62.148	62.579	65.814	68.136	71.769	78.925	92.019	125.94	77
78	47.149	49.787	51.054	54.408	56.087	57.964	59.177	60.101	60.859	61.51	62.083	62.6	63.071	63.506	66.771	69.116	72.786	80.018	93.262	127.61	78
79	47.928	50.592	51.87	55.254	56.948	58.842	60.065	60.998	61.763	62.419	62.998	63.519	63.995	64.434	67.729	70.096	73.803	81.11	94.506	129.27	79
80	48.71	51.397	52.687	56.101	57.81	59.72	60.955	61.895	62.668	63.33	63.914	64.439	64.919	65.363	68.688	71.077	74.82	82.203	95.75	130.94	80
81	49.492	52.204	53.506	56.949	58.673	60.6	61.845	62.794	63.573	64.241	64.83	65.36	65.845	66.292	69.647	72.059	75.838	83.297	96.993	132.6	81
82	50.277	53.012	54.325	57.798	59.537	61.48	62.737	63.693	64.479	65.153	65.747	66.282	66.771	67.222	70.607	73.041	76.856	84.39	98.237	134.27	82
83	51.062	53.822	55.146	58.649	60.403	62.362	63.629	64.594	65.386	66.065	66.665	67.204	67.697	68.152	71.568	74.024	77.874	85.484	99.481	135.93	83
84	51.849	54.633	55.968	59.5	61.269	63.244	64.522	65.495	66.294	66.979	67.583	68.128	68.625	69.084	72.529	75.007	78.893	86.578	100.73	137.6	84
85	52.637	55.445	56.791	60.352	62.135	64.127	65.415	66.396	67.202	67.893	68.503	69.051	69.553	70.016	73.49	75.99	79.912	87.672	101.97	139.26	85
86	53.427	56.258	57.615	61.206	63.003	65.011	66.31	67.299	68.111	68.808	69.423	69.976	70.481	70.948	74.452	76.974	80.932	88.767	103.21	140.93	86
87	54.218	57.072	58.441	62.06	63.872	65.897	67.205	68.202	69.021	69.724	70.343	70.901	71.41	71.881	75.415	77.959	81.952	89.861	104.46	142.6	87
88	55.01	57.887	59.267	62.915	64.742	66.782	68.101	69.106	69.932	70.64	71.264	71.827	72.34	72.815	76.378	78.944	82.972	90.956	105.7	144.26	88
89	55.804	58.704	60.095	63.772	65.612	67.669	68.998	70.011	70.843	71.557	72.186	72.753	73.271	73.749	77.342	79.929	83.993	92.051	106.95	145.93	89
90	56.598	59.526	60.923	64.629	66.484	68.556	69.896	70.917	71.755	72.474	73.109	73.68	74.202	74.684	78.306	80.915	85.014	93.146	108.19	147.59	90
91	57.394	60.344	61.753	65.487	67.356	69.444	70.794	71.823	72.668	73.393	74.032	74.608	75.134	75.62	79.271	81.901	86.035	94.242	109.44	149.26	91
92	58.192	61.164	62.584	66.346	68.229	70.333	71.693	72.73	73.581	74.311	74.956	75.536	76.066	76.556	80.236	82.888	87.057	95.338	110.68	150.92	92
93	58.99	61.985	63.416	67.206	69.103	71.222	72.593	73.637	74.495	75.231	75.88	76.465	76.999	77.493	81.201	83.875	88.079	96.434	111.93	152.59	93
94	59.789	62.807	64.248	68.067	69.978	72.113	73.493	74.545	75.41	76.151	76.805	77.394	77.932	78.43	82.167	84.862	89.101	97.53	113.17	154.26	94
95	60.59	63.63	65.082	68.928	70.853	73.004	74.394	75.454	76.325	77.072	77.731	78.324	78.866	79.368	83.134	85.85	90.123	98.626	114.42	155.92	95
96	61.392	64.454	65.917	69.791	71.729	73.896	75.296	76.364	77.241	77.993	78.657	79.255	79.801	80.306	84.1	86.838	91.146	99.722	115.66	157.59	96
97	62.194	65.279	66.752	70.654	72.606	74.788	76.199	77.274	78.157	78.915	79.584	80.186	80.736	81.245	85.068	87.826	92.169	100.82	116.91	159.25	97
98	62.998	66.105	67.589	71.518	73.484	75.681	77.102	78.185	79.074	79.837	80.511	81.117	81.672	82.184	86.035	88.815	93.193	101.92	118.15	160.92	98
99	63.803	66.932	68.426	72.383	74.363	76.575	78.006	79.096	79.992	80.76	81.439	82.05	82.608	83.124	87.003	89.804	94.216	103.01	119.4	162.59	99
100	64.609	67.76	69.265	73.248	75.242	77.469	78.91	80.008	80.91	81.684	82.367	82.982	83.545	84.064	87.972	90.794	95.24	104.11	120.64	164.25	100
101	65.416	68.589	70.104	74.115	76.122	78.364	79.815	80.92	81.829	82.608	83.296	83.916	84.482	85.005	88.941	91.784	96.265	105.21	121.89	165.92	101
102	66.224	69.419	70.944	74.982	77.003	79.26	80.72	81.833	82.748	83.533	84.225	84.849	85.419	85.946	89.91	92.774	97.289	106.3	123.13	167.58	102
103	67.034	70.249	71.785	75.85	77.884	80.157	81.627	82.747	83.668	84.458	85.155	85.783	86.357	86.888	90.88	93.765	98.314	107.4	124.38	169.25	103
104	67.844	71.081	72.627	76.719	78.766	81.054	82.533	83.661	84.588	85.384	86.086	86.718	87.296	87.83	91.85	94.756	99.339	108.5	125.63	170.91	104
105	68.655	71.913	73.47	77.588	79.649	81.951	83.441	84.576	85.509	86.31	87.017	87.653	88.235	88.773	92.821	95.747	100.36	109.6	126.87	172.58	105
106	69.467	72.747	74.313	78.458	80.532	82.85	84.349	85.492	86.431	87.237	87.948	88.589	89.175	89.716	93.791	96.738	101.39	110.7	128.12	174.25	106
107	70.28	73.581	75.158	79.329	81.416	83.748	85.257	86.407	87.353	88.164	88.88	89.525	90.115	90.66	94.763	97.73	102.42	111.79	129.36	175.91	107
108	71.094	74.416	76.003	80.201	82.301	84.648	86.166	87.324	88.275	89.092	89.812	90.462	91.055	91.604	95.734	98.722	103.44	112.89	130.61	177.58	108
109	71.908	75.252	76.849	81.073	83.186	85.548	87.076	88.241	89.198	90.02	90.745	91.399	91.996	92.548	96.706	99.715	104.47	113.99	131.86	179.24	109
110	72.724	76.089	77.696	81.946	84.072	86.449	87.986	89.158	90.121	90.948	91.678	92.336	92.937	93.493	97.678	100.71	105.49	115.09	133.1	180.91	110
111	73.541	76.926	78.543	82.819	84.959	87.35	88.897	90.076	91.045	91.878	92.612	93.274	93.879	94.438	98.651	101.7	106.52	116.19	134.35	182.58	111
112	74.358	77.764	79.391	83.694	85.846	88.251	89.808	90.994	91.97	92.807	93.546	94.212	94.821	95.384	99.624	102.69	107.55	117.29	135.59	184.24	112
113	75.177	78.604	80.24	84.568	86.734	89.154	90.719	91.913	92.895	93.737	94.481	95.151	95.764	96.33	100.6	103.69	108.57	118.39	136.84	185.91	113
114	75.996	79.443	81.09	85.445	87.622	90.057	91.632	92.833	93.82	94.668	95.416	96.09	96.707	97.277	101.57	104.68	109.6	119.49	138.09	187.57	114
115	76.816	80.284	81.941	86.321	88.511	90.96	92.544	93.753	94.746	95.599	96.352	97.03	97.65	98.223	102.54	105.68	110.63	120.58	139.33	189.24	115
116	77.637	81.126	82.792	87.197	89.401	91.864	93.458	94.673	95.672	96.53	97.287	97.97	98.594	99.171	103.52	106.67	111.66	121.68	140.58	190.91	116
117	78.459	81.968	83.644	88.075	90.291	92.768	94.371	95.594	96.599	97.462	98.224	98.91	99.538	100.12	104.49	107.66	112.69	122.78	141.83	192.57	117
118	79.281	82.811	84.496	88.953	91.181	93.673	95.285	96.515	97.526	98.394	99.16	99.851	100.48	101.07	105.47	108.66	113.71	123.88	143.07	194.24	118
119	80.104	83.654	85.35	89.831	92.073	94.578	96.2	97.437	98.454	99.327	100.1	100.79	101.43	102.01	106.44	109.66	114.74	124.98	144.32	195.91	119
120	80.929	84.499	86.204	90.71	92.964	95.484	97.115	98.359	99.382	100.26	101.04	101.73	102.37	102.96	107.42	110.65	115.77	126.08	145.57	197.57	120
121	81.754	85.344	87.058	91.59	93.857	96.391	98.031	99.282	100.31	101.19	101.97	102.68	103.32	103.91	108.39	111.65	116.8	127.18	146.81	199.24	121
122	82.579	86.189	87.914	92.471	94.75	97.298	98.947	100.2	101.24	102.13	102.91	103.62	104.26	104.86	109.37	112.64	117.83	128.28	148.06	200.9	122
123	83.406	87.036	88.77	93.351	95.643	98.205	99.863	101.13	102.17	103.06	103.85	104.56	105.21	105.81	110.35	113.64	118.86	129.38	149.31	202.57	123
124	84.233	87.883	89.626	94.233	96.537	99.113	100.78	102.05	103.1	104	104.79	105.5	106.16	106.76	111.32	114.64	119.89	130.48	150.55	204.24	124
125	85.061	88.731	90.483	95.115	97.431	100.02	101.7	102.98	104.03	104.93	105.73	106.45	107.1	107.71	112.3	115.63	120.92	131.58	151.8	205.9	125
126	85.911	89.579	91.341	95.997	98.326	100.93	102.62	103.9	104.96	105.87	106.67	107.39	108.05	108.66	113.28	116.63	121.95	132.68	153.05	207.57	126
127	86.74	90.428	92.2	96.881	99.222	101.84	103.53	104.83	105.89	106.8	107.61	108.34	109	109.61	114.25	117.63	122.98	133.78	154.29	209.23	127
128	87.57	91.278	93.059	97.764	100.12	102.75	104.45	105.75	106.82	107.74	108.55	109.28	109.95	110.57	115.23	118.62	124.01	134.88	155.54	210.9	128
129	88.4	92.129	93.919	98.648	101.01	103.66	105.37	106.68	107.75	108.67	109.49	110.22	110.9	111.52	116.21	119.62	125.04	135.99	156.79	212.57	129
130	89.232	92.98	94.779	99.533	101.91	104.57	106.29	107.6	108.68	109.61	110.43	111.17	111.85	112.47	117.19	120.62	126.07	137.09	158.03	214.23	130
131	90.064	93.831	95.64	100.42	102.81	105.48	107.21	108.53	109.62	110.55	111.37	112.12	112.79	113.42	118.17	121.62	127.1	138.19	159.28	215.9	131
132	90.896	94.684	96.502	101.3	103.71	106.39	108.13	109.46	110.55	111.49	112.31	113.06	113.74	114.38	119.15	122.62	128.13	139.29	160.53	217.57	132
133	91.73	95.537	97.364	102.19	104.6	107.3	109.05	110.39	111.48	112.42	113.26	114.01	114.69	115.33	120.12	123.61	129.16	140.39	161.77	219.23	133
134	92.564	96.39	98.226	103.08	105.5	108.22	109.97	111.31	112.42	113.36	114.2	114.95	115.64	116.28	121.1	124.61	130.19	141.49	163.02	220.9	134
135	93.399	97.244	99.09	103.96	106.4	109.13	110.89	112.24	113.35	114.3	115.14	115.9	116.59	117.24	122.08	125.61	131.22	142.59	164.27	222.56	135
136	94.234	98.099	99.953	104.85	107.3	110.04	111.82	113.17	114.28	115.24	116.09	116.85	117.54	118.19	123.06	126.61	132.25	143.69	165.52	224.23	136
137	95.07	98.954	100.82	105.74	108.2	110.95	112.74	114.1	115.22	116.18	117.03	117.8	118.5	119.14	124.04	127.61	133.28	144.8	166.76	225.9	137
138	95.907	99.81	101.68	106.63	109.1	111.87	113.66	115.03	116.15	117.12	117.97	118.74	119.45	120.1	125.02	128.61	134.32	145.9	168.01	227.56	138
139	96.744	100.67	102.55	107.52	110	112.78	114.58	115.96	117.09	118.06	118.92	119.69	120.4	121.05	126	129.61	135.35	147	169.26	229.23	139
140	97.582	101.52	103.41	108.41	110.9	113.7	115.51	116.89	118.02	119	119.86	120.64	121.35	122.01	126.98	130.61	136.38	148.1	170.5	230.9	140
141	98.421	102.38	104.28	109.3	111.81	114.61	116.43	117.82	118.96	119.94	120.81	121.59	122.3	122.96	127.97	131.61	137.41	149.2	171.75	232.56	141
142	99.26	103.24	105.15	110.19	112.71	115.53	117.35	118.75	119.9	120.88	121.75	122.54	123.26	123.92	128.95	132.61	138.44	150.3	173	234.23	142
143	100.1	104.1	106.02	111.08	113.61	116.44	118.28	119.68	120.83	121.82	122.7	123.49	124.21	124.88	129.93	133.61	139.48	151.41	174.25	235.89	143
144	100.94	104.96	106.88	111.97	114.51	117.36	119.2	120.61	121.77	122.76	123.64	124.44	125.16	125.83	130.91	134.61	140.51	152.51	175.49	237.56	144
145	101.78	105.82	107.75	112.86	115.42	118.28	120.13	121.54	122.71	123.71	124.59	125.39	126.11	126.79	131.89	135.61	141.54	153.61	176.74	239.23	145
146	102.62	106.68	108.62	113.75	116.32	119.19	121.05	122.47	123.64	124.65	125.54	126.34	127.07	127.75	132.87	136.61	142.57	154.71	177.99	240.89	146
147	103.46	107.54	109.49	114.65	117.23	120.11	121.98	123.41	124.58	125.59	126.48	127.29	128.02	128.7	133.86	137.61	143.61	155.82	179.24	242.56	147
148	104.31	108.4	110.36	115.54	118.13	121.03	122.91	124.34	125.52	126.53	127.43	128.24	128.98	129.66	134.84	138.61	144.64	156.92	180.48	244.23	148
149	105.15	109.26	111.23	116.43	119.04	121.95	123.83	125.27	126.46	127.48	128.38	129.19	129.93	130.62	135.82	139.62	145.67	158.02	181.73	245.89	149
150	105.99	110.12	112.1	117.33	119.94	122.86	124.76	126.21	127.4	128.42	129.32	130.14	130.88	131.58	136.8	140.62	146.71	159.12	182.98	247.56	150
151	106.84	110.99	112.97	118.22	120.85	123.78	125.69	127.14	128.33	129.36	130.27	131.09	131.84	132.53	137.79	141.62	147.74	160.23	184.23	249.22	151
152	107.68	111.85	113.85	119.12	121.75	124.7	126.61	128.07	129.27	130.31	131.22	132.04	132.79	133.49	138.77	142.62	148.77	161.33	185.47	250.89	152
153	108.53	112.71	114.72	120.01	122.66	125.62	127.54	129.01	130.21	131.25	132.17	132.99	133.75	134.45	139.75	143.62	149.81	162.43	186.72	252.56	153
154	109.38	113.58	115.59	120.91	123.57	126.54	128.47	129.94	131.15	132.19	133.12	133.95	134.71	135.41	140.74	144.63	150.84	163.53	187.97	254.22	154
155	110.22	114.44	116.46	121.8	124.47	127.46	129.4	130.88	132.09	133.14	134.06	134.9	135.66	136.37	141.72	145.63	151.87	164.64	189.22	255.89	155
156	111.07	115.31	117.34	122.7	125.38	128.38	130.33	131.81	133.03	134.08	135.01	135.85	136.62	137.33	142.7	146.63	152.91	165.74	190.47	257.56	156
157	111.92	116.17	118.21	123.6	126.29	129.3	131.25	132.75	133.97	135.03	135.96	136.8	137.57	138.29	143.69	147.63	153.94	166.84	191.71	259.22	157
158	112.77	117.04	119.09	124.49	127.2	130.22	132.18	133.68	134.91	135.97	136.91	137.76	138.53	139.25	144.67	148.64	154.98	167.95	192.96	260.89	158
159	113.61	117.91	119.96	125.39	128.11	131.14	133.11	134.62	135.86	136.92	137.86	138.71	139.49	140.21	145.66	149.64	156.01	169.05	194.21	262.56	159
160	114.46	118.77	120.84	126.29	129.01	132.07	134.04	135.55	136.8	137.87	138.81	139.66	140.44	141.17	146.64	150.64	157.05	170.15	195.46	264.22	160
161	115.31	119.64	121.71	127.19	129.92	132.99	134.97	136.49	137.74	138.81	139.76	140.62	141.4	142.13	147.63	151.65	158.08	171.25	196.7	265.89	161
162	116.16	120.51	122.59	128.08	130.83	133.91	135.9	137.43	138.68	139.76	140.71	141.57	142.36	143.09	148.61	152.65	159.12	172.36	197.95	267.55	162
163	117.01	121.38	123.47	128.98	131.74	134.83	136.83	138.36	139.62	140.71	141.66	142.53	143.32	144.05	149.6	153.66	160.15	173.46	199.2	269.22	163
164	117.87	122.25	124.35	129.88	132.65	135.75	137.77	139.3	140.57	141.65	142.61	143.48	144.28	145.01	150.58	154.66	161.19	174.56	200.45	270.89	164
165	118.72	123.12	125.22	130.78	133.56	136.68	138.7	140.24	141.51	142.6	143.57	144.44	145.23	145.97	151.57	155.66	162.22	175.67	201.7	272.55	165
166	119.57	123.99	126.1	131.68	134.48	137.6	139.63	141.18	142.45	143.55	144.52	145.39	146.19	146.93	152.55	156.67	163.26	176.77	202.94	274.22	166
167	120.42	124.86	126.98	132.58	135.39	138.52	140.56	142.11	143.39	144.49	145.47	146.35	147.15	147.89	153.54	157.67	164.29	177.88	204.19	275.89	167
168	121.28	125.73	127.86	133.48	136.3	139.45	141.49	143.05	144.34	145.44	146.42	147.3	148.11	148.86	154.53	158.68	165.33	178.98	205.44	277.55	168
169	122.13	126.6	128.74	134.38	137.21	140.37	142.42	143.99	145.28	146.39	147.37	148.26	149.07	149.82	155.51	159.68	166.36	180.08	206.69	279.22	169
170	122.98	127.47	129.62	135.29	138.12	141.3	143.36	144.93	146.23	147.34	148.32	149.21	150.03	150.78	156.5	160.69	167.4	181.19	207.94	280.88	170
171	123.84	128.34	130.5	136.19	139.04	142.22	144.29	145.87	147.17	148.29	149.28	150.17	150.99	151.74	157.48	161.69	168.43	182.29	209.18	282.55	171
172	124.69	129.21	131.38	137.09	139.95	143.15	145.22	146.81	148.11	149.24	150.23	151.13	151.95	152.71	158.47	162.7	169.47	183.39	210.43	284.22	172
173	125.55	130.09	132.26	137.99	140.86	144.07	146.16	147.75	149.06	150.19	151.18	152.08	152.91	153.67	159.46	163.7	170.5	184.5	211.68	285.88	173
174	126.4	130.96	133.14	138.89	141.77	145	147.09	148.69	150	151.14	152.14	153.04	153.87	154.63	160.44	164.71	171.54	185.6	212.93	287.55	174
175	127.26	131.83	134.02	139.8	142.69	145.92	148.02	149.63	150.95	152.08	153.09	154	154.83	155.6	161.43	165.71	172.58	186.71	214.18	289.22	175
176	128.12	132.71	134.9	140.7	143.6	146.85	148.96	150.57	151.89	153.03	154.04	154.95	155.79	156.56	162.42	166.72	173.61	187.81	215.42	290.88	176
177	128.97	133.58	135.79	141.6	144.52	147.78	149.89	151.51	152.84	153.98	155	155.91	156.75	157.52	163.41	167.72	174.65	188.91	216.67	292.55	177
178	129.83	134.46	136.67	142.51	145.43	148.7	150.83	152.45	153.79	154.93	155.95	156.87	157.71	158.49	164.39	168.73	175.69	190.02	217.92	294.22	178
179	130.69	135.33	137.55	143.41	146.35	149.63	151.76	153.39	154.73	155.88	156.91	157.83	158.67	159.45	165.38	169.73	176.72	191.12	219.17	295.88	179
180	131.55	136.21	138.44	144.32	147.26	150.56	152.7	154.33	155.68	156.84	157.86	158.78	159.63	160.42	166.37	170.74	177.76	192.23	220.42	297.55	180
181	132.41	137.08	139.32	145.22	148.18	151.49	153.63	155.27	156.62	157.79	158.81	159.74	160.59	161.38	167.36	171.75	178.79	193.33	221.66	299.22	181
182	133.27	137.96	140.2	146.13	149.09	152.41	154.57	156.21	157.57	158.74	159.77	160.7	161.55	162.34	168.35	172.75	179.83	194.44	222.91	300.88	182
183	134.13	138.84	141.09	147.03	150.01	153.34	155.5	157.16	158.52	159.69	160.72	161.66	162.52	163.31	169.33	173.76	180.87	195.54	224.16	302.55	183
184	134.99	139.71	141.97	147.94	150.93	154.27	156.44	158.1	159.46	160.64	161.68	162.62	163.48	164.27	170.32	174.77	181.91	196.65	225.41	304.21	184
185	135.85	140.59	142.86	148.85	151.84	155.2	157.38	159.04	160.41	161.59	162.64	163.58	164.44	165.24	171.31	175.77	182.94	197.75	226.66	305.88	185
186	136.71	141.47	143.74	149.75	152.76	156.13	158.31	159.98	161.36	162.54	163.59	164.54	165.4	166.21	172.3	176.78	183.98	198.85	227.91	307.55	186
187	137.57	142.35	144.63	150.66	153.68	157.06	159.25	160.93	162.31	163.5	164.55	165.5	166.37	167.17	173.29	177.79	185.02	199.96	229.15	309.21	187
188	138.43	143.22	145.52	151.57	154.59	157.99	160.19	161.87	163.25	164.45	165.5	166.46	167.33	168.14	174.28	178.79	186.05	201.06	230.4	310.88	188
189	139.29	144.1	146.4	152.47	155.51	158.91	161.12	162.81	164.2	165.4	166.46	167.42	168.29	169.1	175.27	179.8	187.09	202.17	231.65	312.55	189
190	140.16	144.98	147.29	153.38	156.43	159.84	162.06	163.76	165.15	166.35	167.42	168.37	169.25	170.07	176.26	180.81	188.13	203.27	232.9	314.21	190
191	141.02	145.86	148.18	154.29	157.35	160.77	163	164.7	166.1	167.31	168.37	169.34	170.22	171.03	177.25	181.81	189.17	204.38	234.15	315.88	191
192	141.88	146.74	149.07	155.2	158.27	161.7	163.94	165.64	167.05	168.26	169.33	170.3	171.18	172	178.24	182.82	190.2	205.48	235.4	317.55	192
193	142.75	147.62	149.96	156.11	159.19	162.64	164.87	166.59	168	169.21	170.29	171.26	172.14	172.97	179.23	183.83	191.24	206.59	236.64	319.21	193
194	143.61	148.5	150.85	157.01	160.1	163.57	165.81	167.53	168.95	170.16	171.24	172.22	173.11	173.93	180.22	184.84	192.28	207.69	237.89	320.88	194
195	144.48	149.38	151.73	157.92	161.02	164.5	166.75	168.48	169.9	171.12	172.2	173.18	174.07	174.9	181.21	185.85	193.32	208.8	239.14	322.55	195
196	145.34	150.26	152.62	158.83	161.94	165.43	167.69	169.42	170.85	172.07	173.16	174.14	175.04	175.87	182.2	186.85	194.35	209.9	240.39	324.21	196
197	146.21	151.15	153.51	159.74	162.86	166.36	168.63	170.36	171.79	173.03	174.12	175.1	176	176.84	183.19	187.86	195.39	211.01	241.64	325.88	197
198	147.07	152.03	154.4	160.65	163.78	167.29	169.57	171.31	172.74	173.98	175.07	176.06	176.96	177.8	184.18	188.87	196.43	212.11	242.89	327.54	198
199	147.94	152.91	155.29	161.56	164.7	168.22	170.51	172.25	173.69	174.93	176.03	177.02	177.93	178.77	185.17	189.88	197.47	213.22	244.13	329.21	199
200	148.8	153.79	156.18	162.47	165.62	169.15	171.45	173.2	174.65	175.89	176.99	177.98	178.89	179.74	186.16	190.89	198.51	214.32	245.38	330.88	200
201	149.67	154.68	157.07	163.38	166.54	170.09	172.39	174.15	175.6	176.84	177.95	178.95	179.86	180.71	187.15	191.89	199.55	215.43	246.63	332.54	201
202	150.54	155.56	157.96	164.29	167.47	171.02	173.33	175.09	176.55	177.8	178.91	179.91	180.82	181.67	188.14	192.9	200.58	216.53	247.88	334.21	202
203	151.41	156.44	158.85	165.2	168.39	171.95	174.27	176.04	177.5	178.75	179.87	180.87	181.79	182.64	189.13	193.91	201.62	217.64	249.13	335.88	203
204	152.27	157.33	159.74	166.12	169.31	172.88	175.21	176.98	178.45	179.71	180.82	181.83	182.75	183.61	190.12	194.92	202.66	218.74	250.38	337.54	204
205	153.14	158.21	160.64	167.03	170.23	173.82	176.15	177.93	179.4	180.66	181.78	182.79	183.72	184.58	191.11	195.93	203.7	219.85	251.63	339.21	205
206	154.01	159.09	161.53	167.94	171.15	174.75	177.09	178.88	180.35	181.62	182.74	183.76	184.69	185.55	192.1	196.94	204.74	220.95	252.87	340.88	206
207	154.88	159.98	162.42	168.85	172.07	175.68	178.03	179.82	181.3	182.57	183.7	184.72	185.65	186.52	193.1	197.95	205.78	222.06	254.12	342.54	207
208	155.75	160.86	163.31	169.76	173	176.62	178.97	180.77	182.25	183.53	184.66	185.68	186.62	187.48	194.09	198.96	206.82	223.17	255.37	344.21	208
209	156.62	161.75	164.21	170.68	173.92	177.55	179.91	181.72	183.21	184.49	185.62	186.65	187.58	188.45	195.08	199.97	207.85	224.27	256.62	345.88	209
210	157.49	162.64	165.1	171.59	174.84	178.49	180.85	182.66	184.16	185.44	186.58	187.61	188.55	189.42	196.07	200.97	208.89	225.38	257.87	347.54	210
211	158.36	163.52	165.99	172.5	175.77	179.42	181.8	183.61	185.11	186.4	187.54	188.57	189.52	190.39	197.06	201.98	209.93	226.48	259.12	349.21	211
212	159.23	164.41	166.89	173.42	176.69	180.36	182.74	184.56	186.06	187.36	188.5	189.54	190.48	191.36	198.06	202.99	210.97	227.59	260.37	350.88	212
213	160.1	165.29	167.78	174.33	177.61	181.29	183.68	185.51	187.01	188.31	189.46	190.5	191.45	192.33	199.05	204	212.01	228.69	261.61	352.54	213
214	160.97	166.18	168.67	175.24	178.54	182.22	184.62	186.46	187.97	189.27	190.42	191.46	192.42	193.3	200.04	205.01	213.05	229.8	262.86	354.21	214
215	161.84	167.07	169.57	176.16	179.46	183.16	185.56	187.4	188.92	190.23	191.38	192.43	193.38	194.27	201.03	206.02	214.09	230.9	264.11	355.87	215
216	162.71	167.96	170.46	177.07	180.38	184.1	186.51	188.35	189.87	191.18	192.34	193.39	194.35	195.24	202.02	207.03	215.13	232.01	265.36	357.54	216
217	163.59	168.84	171.36	177.99	181.31	185.03	187.45	189.3	190.83	192.14	193.3	194.35	195.32	196.21	203.02	208.04	216.17	233.12	266.61	359.21	217
218	164.46	169.73	172.25	178.9	182.23	185.97	188.39	190.25	191.78	193.1	194.26	195.32	196.29	197.18	204.01	209.05	217.21	234.22	267.86	360.87	218
219	165.33	170.62	173.15	179.82	183.16	186.9	189.34	191.2	192.73	194.05	195.23	196.28	197.25	198.15	205	210.06	218.25	235.33	269.11	362.54	219
220	166.2	171.51	174.05	180.73	184.08	187.84	190.28	192.15	193.69	195.01	196.19	197.25	198.22	199.12	206	211.07	219.29	236.43	270.36	364.21	220
221	167.08	172.4	174.94	181.65	185.01	188.77	191.22	193.1	194.64	195.97	197.15	198.21	199.19	200.09	206.99	212.08	220.33	237.54	271.6	365.87	221
222	167.95	173.29	175.84	182.56	185.93	189.71	192.17	194.04	195.59	196.93	198.11	199.18	200.16	201.06	207.98	213.09	221.37	238.65	272.85	367.54	222
223	168.83	174.18	176.73	183.48	186.86	190.65	193.11	194.99	196.55	197.89	199.07	200.14	201.12	202.04	208.97	214.1	222.41	239.75	274.1	369.21	223
224	169.7	175.07	177.63	184.39	187.78	191.58	194.05	195.94	197.5	198.85	200.03	201.11	202.09	203.01	209.97	215.11	223.45	240.86	275.35	370.87	224
225	170.57	175.96	178.53	185.31	188.71	192.52	195	196.89	198.46	199.8	201	202.07	203.06	203.98	210.96	216.12	224.48	241.96	276.6	372.54	225
226	171.45	176.85	179.43	186.23	189.64	193.46	195.94	197.84	199.41	200.76	201.96	203.04	204.03	204.95	211.95	217.14	225.52	243.07	277.85	374.21	226
227	172.32	177.74	180.32	187.14	190.56	194.4	196.89	198.79	200.37	201.72	202.92	204	205	205.92	212.95	218.15	226.56	244.18	279.1	375.87	227
228	173.2	178.63	181.22	188.06	191.49	195.33	197.83	199.74	201.32	202.68	203.88	204.97	205.97	206.89	213.94	219.16	227.6	245.28	280.35	377.54	228
229	174.08	179.52	182.12	188.98	192.42	196.27	198.78	200.69	202.28	203.64	204.85	205.94	206.94	207.86	214.94	220.17	228.65	246.39	281.59	379.21	229
230	174.95	180.41	183.02	189.9	193.34	197.21	199.72	201.64	203.23	204.6	205.81	206.9	207.91	208.84	215.93	221.18	229.69	247.49	282.84	380.87	230
231	175.83	181.3	183.92	190.81	194.27	198.15	200.67	202.6	204.19	205.56	206.77	207.87	208.87	209.81	216.92	222.19	230.73	248.6	284.09	382.54	231
232	176.71	182.19	184.82	191.73	195.2	199.09	201.61	203.55	205.14	206.52	207.73	208.83	209.84	210.78	217.92	223.2	231.77	249.71	285.34	384.21	232
233	177.58	183.08	185.71	192.65	196.13	200.02	202.56	204.5	206.1	207.48	208.7	209.8	210.81	211.75	218.91	224.21	232.81	250.81	286.59	385.87	233
234	178.46	183.98	186.61	193.57	197.05	200.96	203.5	205.45	207.05	208.44	209.66	210.77	211.78	212.72	219.91	225.22	233.85	251.92	287.84	387.54	234
235	179.34	184.87	187.51	194.49	197.98	201.9	204.45	206.4	208.01	209.4	210.62	211.73	212.75	213.7	220.9	226.23	234.89	253.02	289.09	389.2	235
236	180.22	185.76	188.41	195.4	198.91	202.84	205.4	207.35	208.97	210.36	211.59	212.7	213.72	214.67	221.9	227.25	235.93	254.13	290.34	390.87	236
237	181.09	186.65	189.31	196.32	199.84	203.78	206.34	208.3	209.92	211.32	212.55	213.67	214.69	215.64	222.89	228.26	236.97	255.24	291.58	392.54	237
238	181.97	187.55	190.21	197.24	200.77	204.72	207.29	209.25	210.88	212.28	213.52	214.64	215.66	216.61	223.88	229.27	238.01	256.34	292.83	394.2	238
239	182.85	188.44	191.11	198.16	201.69	205.66	208.23	210.21	211.83	213.24	214.48	215.6	216.63	217.59	224.88	230.28	239.05	257.45	294.08	395.87	239
240	183.73	189.34	192.02	199.08	202.62	206.6	209.18	211.16	212.79	214.2	215.44	216.57	217.6	218.56	225.87	231.29	240.09	258.56	295.33	397.54	240
241	184.61	190.23	192.92	200	203.55	207.54	210.13	212.11	213.75	215.16	216.41	217.54	218.57	219.53	226.87	232.3	241.13	259.66	296.58	399.2	241
242	185.49	191.12	193.82	200.92	204.48	208.48	211.07	213.06	214.7	216.12	217.37	218.5	219.54	220.51	227.86	233.32	242.17	260.77	297.83	400.87	242
243	186.37	192.02	194.72	201.84	205.41	209.42	212.02	214.02	215.66	217.08	218.34	219.47	220.51	221.48	228.86	234.33	243.21	261.88	299.08	402.54	243
244	187.25	192.91	195.62	202.76	206.34	210.36	212.97	214.97	216.62	218.04	219.3	220.44	221.48	222.45	229.85	235.34	244.25	262.98	300.33	404.2	244
245	188.13	193.81	196.52	203.68	207.27	211.3	213.92	215.92	217.58	219	220.27	221.41	222.46	223.43	230.85	236.35	245.29	264.09	301.58	405.87	245
246	189.01	194.7	197.42	204.6	208.2	212.24	214.86	216.87	218.53	219.96	221.23	222.38	223.43	224.4	231.84	237.36	246.34	265.2	302.82	407.54	246
247	189.89	195.6	198.33	205.52	209.13	213.18	215.81	217.83	219.49	220.92	222.2	223.34	224.4	225.37	232.84	238.38	247.38	266.3	304.07	409.2	247
248	190.77	196.49	199.23	206.44	210.06	214.12	216.76	218.78	220.45	221.89	223.16	224.31	225.37	226.35	233.84	239.39	248.42	267.41	305.32	410.87	248
249	191.65	197.39	200.13	207.37	210.99	215.06	217.71	219.73	221.41	222.85	224.13	225.28	226.34	227.32	234.83	240.4	249.46	268.52	306.57	412.54	249
250	192.53	198.29	201.04	208.29	211.92	216	218.65	220.69	222.36	223.81	225.09	226.25	227.31	228.3	235.83	241.41	250.5	269.62	307.82	414.2	250
251	193.42	199.18	201.94	209.21	212.85	216.94	219.6	221.64	223.32	224.77	226.06	227.22	228.28	229.27	236.82	242.43	251.54	270.73	309.07	415.87	251
252	194.3	200.08	202.84	210.13	213.78	217.88	220.55	222.59	224.28	225.73	227.02	228.19	229.25	230.25	237.82	243.44	252.58	271.84	310.32	417.54	252
253	195.18	200.98	203.75	211.05	214.72	218.83	221.5	223.55	225.24	226.7	227.99	229.16	230.23	231.22	238.81	244.45	253.62	272.94	311.57	419.2	253
254	196.06	201.87	204.65	211.97	215.65	219.77	222.45	224.5	226.2	227.66	228.95	230.12	231.2	232.19	239.81	245.46	254.67	274.05	312.82	420.87	254
255	196.95	202.77	205.55	212.9	216.58	220.71	223.4	225.46	227.16	228.62	229.92	231.09	232.17	233.17	240.81	246.48	255.71	275.16	314.07	422.54	255
256	197.83	203.67	206.46	213.82	217.51	221.65	224.35	226.41	228.11	229.58	230.89	232.06	233.14	234.14	241.8	247.49	256.75	276.26	315.31	424.2	256
257	198.71	204.57	207.36	214.74	218.44	222.59	225.3	227.36	229.07	230.55	231.85	233.03	234.11	235.12	242.8	248.5	257.79	277.37	316.56	425.87	257
258	199.6	205.46	208.27	215.66	219.37	223.54	226.24	228.32	230.03	231.51	232.82	234	235.09	236.09	243.8	249.52	258.83	278.48	317.81	427.53	258
259	200.48	206.36	209.17	216.59	220.31	224.48	227.19	229.27	230.99	232.47	233.78	234.97	236.06	237.07	244.79	250.53	259.87	279.58	319.06	429.2	259
260	201.36	207.26	210.08	217.51	221.24	225.42	228.14	230.23	231.95	233.43	234.75	235.94	237.03	238.04	245.79	251.54	260.91	280.69	320.31	430.87	260
261	202.25	208.16	210.98	218.43	222.17	226.36	229.09	231.18	232.91	234.4	235.72	236.91	238	239.02	246.78	252.56	261.96	281.8	321.56	432.53	261
262	203.13	209.06	211.89	219.36	223.1	227.31	230.04	232.14	233.87	235.36	236.68	237.88	238.98	239.99	247.78	253.57	263	282.9	322.81	434.2	262
263	204.02	209.96	212.79	220.28	224.04	228.25	230.99	233.09	234.83	236.32	237.65	238.85	239.95	240.97	248.78	254.58	264.04	284.01	324.06	435.87	263
264	204.9	210.86	213.7	221.2	224.97	229.19	231.94	234.05	235.79	237.29	238.62	239.82	240.92	241.95	249.77	255.6	265.08	285.12	325.31	437.53	264
265	205.79	211.75	214.61	222.13	225.9	230.14	232.89	235	236.75	238.25	239.58	240.79	241.89	242.92	250.77	256.61	266.12	286.23	326.56	439.2	265
266	206.67	212.65	215.51	223.05	226.83	231.08	233.84	235.96	237.71	239.21	240.55	241.76	242.87	243.9	251.77	257.62	267.17	287.33	327.8	440.87	266
267	207.56	213.55	216.42	223.98	227.77	232.02	234.79	236.92	238.67	240.18	241.52	242.73	243.84	244.87	252.77	258.64	268.21	288.44	329.05	442.53	267
268	208.44	214.45	217.33	224.9	228.7	232.97	235.74	237.87	239.63	241.14	242.49	243.7	244.81	245.85	253.76	259.65	269.25	289.55	330.3	444.2	268
269	209.33	215.35	218.23	225.83	229.64	233.91	236.69	238.83	240.59	242.11	243.45	244.67	245.79	246.82	254.76	260.66	270.29	290.65	331.55	445.87	269
270	210.22	216.26	219.14	226.75	230.57	234.86	237.64	239.78	241.55	243.07	244.42	245.64	246.76	247.8	255.76	261.68	271.33	291.76	332.8	447.53	270
271	211.1	217.16	220.05	227.68	231.5	235.8	238.6	240.74	242.51	244.03	245.39	246.61	247.73	248.78	256.75	262.69	272.38	292.87	334.05	449.2	271
272	211.99	218.06	220.96	228.6	232.44	236.74	239.55	241.7	243.47	245	246.36	247.58	248.71	249.75	257.75	263.71	273.42	293.98	335.3	450.87	272
273	212.88	218.96	221.86	229.53	233.37	237.69	240.5	242.65	244.43	245.96	247.32	248.55	249.68	250.73	258.75	264.72	274.46	295.08	336.55	452.53	273
274	213.76	219.86	222.77	230.45	234.31	238.63	241.45	243.61	245.39	246.93	248.29	249.52	250.66	251.71	259.75	265.73	275.5	296.19	337.8	454.2	274
275	214.65	220.76	223.68	231.38	235.24	239.58	242.4	244.56	246.35	247.89	249.26	250.5	251.63	252.68	260.74	266.75	276.55	297.3	339.05	455.87	275
276	215.54	221.66	224.59	232.3	236.18	240.52	243.35	245.52	247.31	248.86	250.23	251.47	252.6	253.66	261.74	267.76	277.59	298.4	340.3	457.53	276
277	216.43	222.56	225.5	233.23	237.11	241.47	244.3	246.48	248.27	249.82	251.2	252.44	253.58	254.64	262.74	268.78	278.63	299.51	341.54	459.2	277
278	217.32	223.47	226.4	234.16	238.05	242.41	245.26	247.43	249.24	250.79	252.16	253.41	254.55	255.61	263.74	269.79	279.67	300.62	342.79	460.87	278
279	218.2	224.37	227.31	235.08	238.98	243.36	246.21	248.39	250.2	251.75	253.13	254.38	255.53	256.59	264.74	270.8	280.71	301.73	344.04	462.53	279
280	219.09	225.27	228.22	236.01	239.92	244.3	247.16	249.35	251.16	252.72	254.1	255.35	256.5	257.57	265.73	271.82	281.76	302.83	345.29	464.2	280
281	219.98	226.17	229.13	236.93	240.85	245.25	248.11	250.31	252.12	253.68	255.07	256.32	257.48	258.54	266.73	272.83	282.8	303.94	346.54	465.87	281
282	220.87	227.08	230.04	237.86	241.79	246.19	249.06	251.26	253.08	254.65	256.04	257.3	258.45	259.52	267.73	273.85	283.84	305.05	347.79	467.53	282
283	221.76	227.98	230.95	238.79	242.72	247.14	250.02	252.22	254.04	255.61	257.01	258.27	259.42	260.5	268.73	274.86	284.89	306.16	349.04	469.2	283
284	222.65	228.88	231.86	239.72	243.66	248.09	250.97	253.18	255	256.58	257.98	259.24	260.4	261.48	269.73	275.88	285.93	307.26	350.29	470.87	284
285	223.54	229.79	232.77	240.64	244.59	249.03	251.92	254.14	255.97	257.55	258.95	260.21	261.37	262.45	270.72	276.89	286.97	308.37	351.54	472.53	285
286	224.43	230.69	233.68	241.57	245.53	249.98	252.87	255.09	256.93	258.51	259.91	261.18	262.35	263.43	271.72	277.91	288.01	309.48	352.79	474.2	286
287	225.32	231.59	234.59	242.5	246.47	250.92	253.83	256.05	257.89	259.48	260.88	262.16	263.32	264.41	272.72	278.92	289.06	310.58	354.04	475.86	287
288	226.21	232.5	235.5	243.43	247.4	251.87	254.78	257.01	258.85	260.44	261.85	263.13	264.3	265.39	273.72	279.93	290.1	311.69	355.28	477.53	288
289	227.1	233.4	236.41	244.35	248.34	252.82	255.73	257.97	259.82	261.41	262.82	264.1	265.27	266.36	274.72	280.95	291.14	312.8	356.53	479.2	289
290	227.99	234.31	237.32	245.28	249.28	253.76	256.69	258.93	260.78	262.37	263.79	265.07	266.25	267.34	275.72	281.96	292.18	313.91	357.78	480.86	290
291	228.88	235.21	238.23	246.21	250.21	254.71	257.64	259.88	261.74	263.34	264.76	266.05	267.23	268.32	276.72	282.98	293.23	315.01	359.03	482.53	291
292	229.77	236.11	239.14	247.14	251.15	255.66	258.59	260.84	262.7	264.31	265.73	267.02	268.2	269.3	277.71	283.99	294.27	316.12	360.28	484.2	292
293	230.66	237.02	240.06	248.07	252.09	256.6	259.55	261.8	263.67	265.27	266.7	267.99	269.18	270.28	278.71	285.01	295.31	317.23	361.53	485.86	293
294	231.56	237.92	240.97	248.99	253.02	257.55	260.5	262.76	264.63	266.24	267.67	268.96	270.15	271.25	279.71	286.02	296.36	318.34	362.78	487.53	294
295	232.45	238.83	241.88	249.92	253.96	258.5	261.45	263.72	265.59	267.21	268.64	269.94	271.13	272.23	280.71	287.04	297.4	319.44	364.03	489.2	295
296	233.34	239.74	242.79	250.85	254.9	259.44	262.41	264.68	266.55	268.17	269.61	270.91	272.1	273.21	281.71	288.05	298.44	320.55	365.28	490.86	296
297	234.23	240.64	243.7	251.78	255.84	260.39	263.36	265.64	267.52	269.14	270.58	271.88	273.08	274.19	282.71	289.07	299.49	321.66	366.53	492.53	297
298	235.12	241.55	244.61	252.71	256.77	261.34	264.31	266.6	268.48	270.11	271.55	272.86	274.06	275.17	283.71	290.09	300.53	322.77	367.78	494.2	298
299	236.02	242.45	245.53	253.64	257.71	262.29	265.27	267.55	269.44	271.07	272.52	273.83	275.03	276.15	284.71	291.1	301.57	323.88	369.03	495.86	299
300	236.91	243.36	246.44	254.57	258.65	263.23	266.22	268.51	270.41	272.04	273.49	274.8	276.01	277.13	285.71	292.12	302.62	324.98	370.28	497.53	300
301	237.8	244.26	247.35	255.5	259.59	264.18	267.18	269.47	271.37	273.01	274.46	275.78	276.98	278.1	286.71	293.13	303.66	326.09	371.52	499.2	301
